//Unit Tests for Mifare Classic
//(C) TechnicallyObsolete 2025
#include "unity.h"
#include "error_codes.h"

#include "rf/rfid/tag_mifare_classic.h"
#include "utils/syslog.h"

#include "expect.h"

#include <stdint.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
void put_ch(unsigned char c){
  putc(c, stderr);
}

bool debugMode = false;

uint32_t demoCtxPtr = 0;
uint32_t const expectedFlagValue = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY;
uint32_t const expectedFlagHaltValue = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY | RFID_GENERIC_FLAG_NO_REPLY;
uint32_t const expectedFlagWriteValue = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY | RFID_GENERIC_FLAG_NO_RETRY;

//To avoid having to re-authenticate on all of the other unit tests,
// store a known good LFSR state here
// This one is after authenticating block 4 with key A
uint64_t const authLfsrBlk4Value = 0xA66D0DD5272A;


bool expectingHalt = false;
bool expectingWrite = false;

uint16_t expectedDataTxBuf[4][128] = {0};
uint16_t dataRxBuf[4][128] = {0};
int expectedDataTxBufWaiting[4] = {0};
int dataRxBufToReturn[4] = {0};
int dataRxStsToReturn[4] = {0};
uint32_t dataRxRecvSts[4] = {0};

int currentRfidIdx = 0;

int rfidTrx(void *ctx, uint32_t flags, uint16_t const * sendBuf, uint_least16_t sendBufLen, uint16_t *recvBuf, uint_least16_t recvBufLen, uint32_t *recvSts){
  TEST_ASSERT_EQUAL_PTR_MESSAGE(&demoCtxPtr, ctx, "Didn't set ctx ptr");

  //Mifare Classic always runs with external parity bits and no CRC,
  // so check that is provided here
  if (expectingHalt){
    TEST_ASSERT_EQUAL_HEX32_MESSAGE(expectedFlagHaltValue, flags, "Didn't set haltFlags correctly");
  } else if (expectingWrite){
    TEST_ASSERT_EQUAL_HEX32_MESSAGE(expectedFlagWriteValue, flags, "Didn't set writeFlags correctly");
  } else {
    TEST_ASSERT_EQUAL_HEX32_MESSAGE(expectedFlagValue, flags, "Didn't set flags correctly");
  }

  //Check the data provided
  TEST_ASSERT_EQUAL_INT_MESSAGE(expectedDataTxBufWaiting[currentRfidIdx], sendBufLen, "Didn't set sendBufLen");
  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedDataTxBuf[currentRfidIdx], sendBuf, sendBufLen, "Didn't set sendBuf");

  (void) recvBuf;
  (void) recvBufLen;
  (void) recvSts;
  if (debugMode){
    printf("TX %d -> ", sendBufLen);
    for(uint_least16_t i = 0; i < sendBufLen; i++){
      printf("%04X ", sendBuf[i]);
    }
    printf("\n");
  }


  if (dataRxBufToReturn[currentRfidIdx] > 0){
    if (debugMode){
      printf("RX %d <- ", dataRxBufToReturn[currentRfidIdx]);
    }
    for(uint_least16_t i = 0; i < dataRxBufToReturn[currentRfidIdx]; i++){
      recvBuf[i] = dataRxBuf[currentRfidIdx][i];

      if (debugMode){
        printf("%04X ", recvBuf[i]);
      }
    }

    if (debugMode){
      printf("\n");
    }
  }

  *recvSts = dataRxRecvSts[currentRfidIdx];

  expectedDataTxBufWaiting[currentRfidIdx] = 0;
  int returnVal = dataRxStsToReturn[currentRfidIdx];

  currentRfidIdx++;
  return returnVal;
}

void test_tag_mifare_classic_auth(void){
  uint32_t uid = 0x8972DB80;
  uint64_t key = 0xFFFFFFFFFFFF;
  uint32_t readerChallenge = 0x99999999;

  uint8_t blockAddr = 4;

  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, &demoCtxPtr, rfidTrx);


  expectedDataTxBufWaiting[0] = 4;
  expectedDataTxBuf[0][0] = 0x8060;
  expectedDataTxBuf[0][1] = 0x0004;
  expectedDataTxBuf[0][2] = 0x80D1;
  expectedDataTxBuf[0][3] = 0x003D;

  dataRxStsToReturn[0] = 4;
  dataRxBufToReturn[0] = 4;
  dataRxBuf[0][0] = 0x8000;
  dataRxBuf[0][1] = 0x8090;
  dataRxBuf[0][2] = 0x0080;
  dataRxBuf[0][3] = 0x00A2;

  //Auth 2nd stage
  expectedDataTxBufWaiting[1] = 8;
  expectedDataTxBuf[1][0] = 0x0070;
  expectedDataTxBuf[1][1] = 0x8006;
  expectedDataTxBuf[1][2] = 0x807F;
  expectedDataTxBuf[1][3] = 0x8029;
  expectedDataTxBuf[1][4] = 0x0071;
  expectedDataTxBuf[1][5] = 0x002F;
  expectedDataTxBuf[1][6] = 0x0005;
  expectedDataTxBuf[1][7] = 0x8056;
  dataRxStsToReturn[1] = 4;
  dataRxBufToReturn[1] = 4;
  dataRxBuf[1][0] = 0x80D9;
  dataRxBuf[1][1] = 0x0039;
  dataRxBuf[1][2] = 0x0015;
  dataRxBuf[1][3] = 0x80D6;

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_authBlock(&tag, key, TAG_MIFARE_CLASSIC_KEY_A, readerChallenge, uid, blockAddr), "Didn't get ERR_OK to auth 1");
}

void test_tag_mifare_classic_block_read(void){
  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, &demoCtxPtr, rfidTrx);
  tag.crypto1Ctx.lfsr = authLfsrBlk4Value;

  expectedDataTxBufWaiting[0] = 4;
  expectedDataTxBuf[0][0] = 0x80BC;
  expectedDataTxBuf[0][1] = 0x00F6;
  expectedDataTxBuf[0][2] = 0x0062;
  expectedDataTxBuf[0][3] = 0x00FA;

  dataRxStsToReturn[0] = 18;
  dataRxBufToReturn[0] = 18;
  dataRxBuf[0][ 0] = 0x003E;
  dataRxBuf[0][ 1] = 0x8083;
  dataRxBuf[0][ 2] = 0x0015;
  dataRxBuf[0][ 3] = 0x80F3;
  dataRxBuf[0][ 4] = 0x80B3;
  dataRxBuf[0][ 5] = 0x0012;
  dataRxBuf[0][ 6] = 0x802A;
  dataRxBuf[0][ 7] = 0x809B;
  dataRxBuf[0][ 8] = 0x0004;
  dataRxBuf[0][ 9] = 0x801B;
  dataRxBuf[0][10] = 0x0097;
  dataRxBuf[0][11] = 0x8031;
  dataRxBuf[0][12] = 0x8058;
  dataRxBuf[0][13] = 0x809A;
  dataRxBuf[0][14] = 0x806D;
  dataRxBuf[0][15] = 0x80E8;
  dataRxBuf[0][16] = 0x80C3;
  dataRxBuf[0][17] = 0x80D5;

  uint8_t dataBack[16] = {0};
  uint8_t const expectedDataBackBlk4[16] = {
    0x11, 0x22, 0x33, 0x44,
    0x55, 0x60, 0x77, 0x97,
    0x99, 0xAA, 0xBB, 0xCC,
    0x00, 0xEE, 0xFF, 0x10
  };

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_blockRead(&tag, 4, dataBack), "Didn't get ERR_OK to read 1");
  TEST_ASSERT_EQUAL_HEX8_ARRAY_MESSAGE(expectedDataBackBlk4, dataBack, 16, "Didn't set databack 1");


  uint8_t const expectedDataBackBlk5[16] = {
    0xFF, 0xEE, 0xDD, 0xCC,
    0xBB, 0xAA, 0x99, 0x88,
    0x77, 0x66, 0x55, 0x44,
    0x33, 0x22, 0x11, 0x00
  };

  expectedDataTxBufWaiting[1] = 4;
  expectedDataTxBuf[1][0] = 0x8064;
  expectedDataTxBuf[1][1] = 0x8051;
  expectedDataTxBuf[1][2] = 0x8015;
  expectedDataTxBuf[1][3] = 0x80B1;

  dataRxStsToReturn[1] = 18;
  dataRxBufToReturn[1] = 18;
  dataRxBuf[1][ 0] = 0x80BB;
  dataRxBuf[1][ 1] = 0x805C;
  dataRxBuf[1][ 2] = 0x80AB;
  dataRxBuf[1][ 3] = 0x00E4;
  dataRxBuf[1][ 4] = 0x0026;
  dataRxBuf[1][ 5] = 0x80C9;
  dataRxBuf[1][ 6] = 0x8075;
  dataRxBuf[1][ 7] = 0x0092;
  dataRxBuf[1][ 8] = 0x80C8;
  dataRxBuf[1][ 9] = 0x001E;
  dataRxBuf[1][10] = 0x00AE;
  dataRxBuf[1][11] = 0x80EB;
  dataRxBuf[1][12] = 0x8075;
  dataRxBuf[1][13] = 0x0058;
  dataRxBuf[1][14] = 0x0076;
  dataRxBuf[1][15] = 0x80A1;
  dataRxBuf[1][16] = 0x80CA;
  dataRxBuf[1][17] = 0x0047;

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_blockRead(&tag, 5, dataBack), "Didn't get ERR_OK to read 2");
  TEST_ASSERT_EQUAL_HEX8_ARRAY_MESSAGE(expectedDataBackBlk5, dataBack, 16, "Didn't set databack 2");
}

void test_tag_mifare_classic_safe_write(void){
  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, &demoCtxPtr, rfidTrx);
  tag.crypto1Ctx.lfsr = authLfsrBlk4Value;

  expectingWrite = true;
  expectedDataTxBufWaiting[0] = 4;
  expectedDataTxBuf[0][0] = 0x802C;
  expectedDataTxBuf[0][1] = 0x00F6;
  expectedDataTxBuf[0][2] = 0x803F;
  expectedDataTxBuf[0][3] = 0x80E3;

  expectedDataTxBufWaiting[1] = 18;
  expectedDataTxBuf[1][ 0] = 0x8003;
  expectedDataTxBuf[1][ 1] = 0x8048;
  expectedDataTxBuf[1][ 2] = 0x0041;
  expectedDataTxBuf[1][ 3] = 0x802F;
  expectedDataTxBuf[1][ 4] = 0x007B;
  expectedDataTxBuf[1][ 5] = 0x00B7;
  expectedDataTxBuf[1][ 6] = 0x80B2;
  expectedDataTxBuf[1][ 7] = 0x8047;
  expectedDataTxBuf[1][ 8] = 0x0080;
  expectedDataTxBuf[1][ 9] = 0x8061;
  expectedDataTxBuf[1][10] = 0x0069;
  expectedDataTxBuf[1][11] = 0x0043;
  expectedDataTxBuf[1][12] = 0x0045;
  expectedDataTxBuf[1][13] = 0x00C9;
  expectedDataTxBuf[1][14] = 0x0076;
  expectedDataTxBuf[1][15] = 0x804F;
  expectedDataTxBuf[1][16] = 0x8055;
  expectedDataTxBuf[1][17] = 0x00BD;

  uint8_t const dataToWrite[16] = {
    0x11, 0x22, 0x33, 0x44,
    0x55, 0x60, 0x77, 0x97,
    0x99, 0xAA, 0xBB, 0xCC,
    0x00, 0xEE, 0xFF, 0x10
  };

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_safeBlockWrite(&tag, 4, dataToWrite), "Didn't get ERR_OK to write");

  //It's easy to make an error handling the ACK bits through crypto1, so check that the LFSR was left in the correct state
  uint64_t expectedLfsrAfterWrite = 0x80689F97A88A;
  TEST_ASSERT_EQUAL_HEX64_MESSAGE(expectedLfsrAfterWrite, tag.crypto1Ctx.lfsr, "Write didn't update LFSR correctly");
}

void test_tag_mifare_classic_halt(void){
  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, &demoCtxPtr, rfidTrx);
  tag.crypto1Ctx.lfsr = 0x80689F97A88A; //Just after a write to block 4

  expectedDataTxBufWaiting[0] = 4;
  expectedDataTxBuf[0][0] = 0x8004;
  expectedDataTxBuf[0][1] = 0x80BA;
  expectedDataTxBuf[0][2] = 0x0019;
  expectedDataTxBuf[0][3] = 0x0089;
  expectingHalt = true;

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_halt(&tag, true), "Didn't get ERR_OK to halt (auth)");
}

void test_tag_mifare_classic_command_chain_test(void){
  //Tests a full Write/Read/Halt sequence
  expectingWrite = true;
  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, &demoCtxPtr, rfidTrx);
  tag.crypto1Ctx.lfsr = authLfsrBlk4Value;

  expectingWrite = true;
  expectedDataTxBufWaiting[0] = 4;
  expectedDataTxBuf[0][0] = 0x802C;
  expectedDataTxBuf[0][1] = 0x00F6;
  expectedDataTxBuf[0][2] = 0x803F;
  expectedDataTxBuf[0][3] = 0x80E3;

  expectedDataTxBufWaiting[1] = 18;
  expectedDataTxBuf[1][ 0] = 0x0013;
  expectedDataTxBuf[1][ 1] = 0x0068;
  expectedDataTxBuf[1][ 2] = 0x0071;
  expectedDataTxBuf[1][ 3] = 0x006F;
  expectedDataTxBuf[1][ 4] = 0x002B;
  expectedDataTxBuf[1][ 5] = 0x00D1;
  expectedDataTxBuf[1][ 6] = 0x00C2;
  expectedDataTxBuf[1][ 7] = 0x80D8;
  expectedDataTxBuf[1][ 8] = 0x0010;
  expectedDataTxBuf[1][ 9] = 0x80C1;
  expectedDataTxBuf[1][10] = 0x80D9;
  expectedDataTxBuf[1][11] = 0x0083;
  expectedDataTxBuf[1][12] = 0x8048;
  expectedDataTxBuf[1][13] = 0x8029;
  expectedDataTxBuf[1][14] = 0x0086;
  expectedDataTxBuf[1][15] = 0x804F;
  expectedDataTxBuf[1][16] = 0x00AD;
  expectedDataTxBuf[1][17] = 0x0059;

  uint8_t const dataToWrite[16] = {
    0x01, 0x02, 0x03, 0x04,
    0x05, 0x06, 0x07, 0x08,
    0x09, 0x0A, 0x0B, 0x0C,
    0x0D, 0x0E, 0x0F, 0x10
  };

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_safeBlockWrite(&tag, 4, dataToWrite), "Didn't get ERR_OK to write");

  expectingWrite = false;
  expectedDataTxBufWaiting[2] = 4;
  expectedDataTxBuf[2][0] = 0x8064;
  expectedDataTxBuf[2][1] = 0x00BE;
  expectedDataTxBuf[2][2] = 0x0068;
  expectedDataTxBuf[2][3] = 0x80AA;

  dataRxStsToReturn[2] = 18;
  dataRxBufToReturn[2] = 18;
  dataRxBuf[2][ 0] = 0x00B3;
  dataRxBuf[2][ 1] = 0x0074;
  dataRxBuf[2][ 2] = 0x002B;
  dataRxBuf[2][ 3] = 0x8099;
  dataRxBuf[2][ 4] = 0x8066;
  dataRxBuf[2][ 5] = 0x80EA;
  dataRxBuf[2][ 6] = 0x801D;
  dataRxBuf[2][ 7] = 0x00B7;
  dataRxBuf[2][ 8] = 0x0071;
  dataRxBuf[2][ 9] = 0x00F1;
  dataRxBuf[2][10] = 0x00A4;
  dataRxBuf[2][11] = 0x804A;
  dataRxBuf[2][12] = 0x8077;
  dataRxBuf[2][13] = 0x8069;
  dataRxBuf[2][14] = 0x80AE;
  dataRxBuf[2][15] = 0x0016;
  dataRxBuf[2][16] = 0x80A2;
  dataRxBuf[2][17] = 0x80AA;

  uint8_t dataBack[16] = {0};

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_blockRead(&tag, 4, dataBack), "Didn't get ERR_OK to read");
  TEST_ASSERT_EQUAL_HEX8_ARRAY_MESSAGE(dataToWrite, dataBack, 16, "Didn't set databack");

  expectingHalt = true;
  expectedDataTxBufWaiting[3] = 4;
  expectedDataTxBuf[3][0] = 0x007A;
  expectedDataTxBuf[3][1] = 0x0017;
  expectedDataTxBuf[3][2] = 0x00E4;
  expectedDataTxBuf[3][3] = 0x8031;

  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, tagMifareClassic_halt(&tag, true), "Didn't get ERR_OK to halt (auth)");
}

void setUp(void){
  currentRfidIdx = 0;
  expectingHalt = false;
  expectingWrite = false;
}

void tearDown(void){
}


int main(void){
//  syslog_init(NULL, SYSLOG_LEVEL_VERBOSE, put_ch);
  UNITY_BEGIN();
  RUN_TEST(test_tag_mifare_classic_auth);
  RUN_TEST(test_tag_mifare_classic_block_read);
  RUN_TEST(test_tag_mifare_classic_safe_write);
  RUN_TEST(test_tag_mifare_classic_halt);
  RUN_TEST(test_tag_mifare_classic_command_chain_test);
  return UNITY_END();

}


