//Unit Tests for CRYPTO1
//(C) TechnicallyObsolete 2025
#include "unity.h"
#include "error_codes.h"

#include "utils/syslog.h"
#include "rf/rfid/mifare_crypto1.h"
#include "expect.h"

#include <stdint.h>


void put_ch(unsigned char c){
  putc(c, stderr);
}

//Basic test for Auth
// this test uses symmetrical values for key and rc,
//  so that if there's a byte swap missing this test will still pass
// (makes it easier to debug the encryption process step-by-step)
void test_mifare_crypto1_auth_sym_key_sym_rc(void){
  uint64_t key = 0xFFFFFFFFFFFFULL;
  uint32_t uid = 0x8972DB80;
  uint32_t tc = 0x009080A2;
  uint32_t rc = 0x99999999;
  uint32_t encAt = 0xD93915d6;
  uint16_t tempOut[8] = {0};

  mifareCrypto1_ctx_t ctx = {0};


  //Check auth with OK key works
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderInit(&ctx, key, uid, tc, rc, tempOut), "Didn't get ERR_OK to authReaderInit 1");

  uint16_t const expectedTempOut[8] = {
    0x0070, 0x8006, 0x807F, 0x8029, 0x0071, 0x002F, 0x0005, 0x8056
  };

  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedTempOut, tempOut, 8, "Didn't set Reader -> Tag reply correctly 1");
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_OK to authReaderHandleTagReply 1");

  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0xFF2F6DBD, ctx.ks[0], "ks0 not set correctly (1)");
  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0xE99FE6B0, ctx.ks[1], "ks1 not set correctly (1)");

  //Check that a failed auth returns DATAERROR, &ctx->ks[1]
  mifareCrypto1_authReaderInit(&ctx, key, uid + 1, tc, rc, tempOut),
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_DATAERROR, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_DATAERROR to authReaderHandleTagReply 2");
}


//Auth1 test, but with non-symmetrical key (still symmetrical rc)
void test_mifare_crypto1_auth_2(void){
  uint64_t key = 0xA0A1A2A3A4A5;
  uint32_t uid = 0x8972DB80;
  uint32_t tc = 0x009080A2;
  uint32_t rc = 0x99999999;
  uint32_t encAt = 0x5447EDC6;
  uint16_t tempOut[8] = {0};

  mifareCrypto1_ctx_t ctx = {0};


  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderInit(&ctx, key, uid, tc, rc, tempOut), "Didn't get ERR_OK to authReaderInit 1");

  uint16_t const expectedTempOut[8] = {0x0023, 0x0054, 0x801E, 0x00C1, 0x0056, 0x8067, 0x8084, 0x00EB};
  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedTempOut, tempOut, 8, "Didn't set Reader -> Tag reply correctly 1");
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_OK to authReaderHandleTagReply 1");

  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x5A75C98B, ctx.ks[0], "ks0 not set correctly (1)");
  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0xBACD8758, ctx.ks[1], "ks1 not set correctly (1)");
}

//Auth1 test, non-symmetrical key and rc
void test_mifare_crypto1_auth_non_sym_key_sym_rc(void){
  uint64_t key = 0xA0A1A2A3A4A5;
  uint32_t uid = 0x8972DB80;
  uint32_t tc = 0x009080A2;
  uint32_t rc = 0x99999999;
  uint32_t encAt = 0x5447EDC6;
  uint16_t tempOut[8] = {0};

  mifareCrypto1_ctx_t ctx = {0};


  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderInit(&ctx, key, uid, tc, rc, tempOut), "Didn't get ERR_OK to authReaderInit 1");

  uint16_t const expectedTempOut[8] = {0x0023, 0x0054, 0x801E, 0x00C1, 0x0056, 0x8067, 0x8084, 0x00EB};
  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedTempOut, tempOut, 8, "Didn't set Reader -> Tag reply correctly 1");
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_OK to authReaderHandleTagReply 1");

  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x5A75C98B, ctx.ks[0], "ks0 not set correctly (1)");
  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0xBACD8758, ctx.ks[1], "ks1 not set correctly (1)");
}

void test_mifare_crypto1_auth_non_sym_key_non_sym_rc(void){
  uint64_t key    = 0xA0A1A2A3A4A5;
  uint32_t uid    = 0x8972DB80;
  uint32_t tc     = 0x009080A2;
  uint32_t rc     = 0xABCD1234;
  uint32_t encAt  = 0x83E3BAB4;
  uint16_t tempOut[8] = {0};

  mifareCrypto1_ctx_t ctx = {0};


  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderInit(&ctx, key, uid, tc, rc, tempOut), "Didn't get ERR_OK to authReaderInit 1");

  uint16_t const expectedTempOut[8] = {0x80EF, 0x00FE, 0x0092, 0x8013, 0x809E, 0x8064, 0x8092, 0x8073};
  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedTempOut, tempOut, 8, "Didn't set Reader -> Tag reply correctly 1");
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_OK to authReaderHandleTagReply 1");

  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x5A75C98B, ctx.ks[0], "ks0 not set correctly (1)");
  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x3A4DDA3F, ctx.ks[1], "ks1 not set correctly (1)");
}

void test_mifare_crypto1_auth_diff_uid(void){
  uint64_t key = 0x204752454154;
  uint32_t uid = 0xCBBD0A53;
  uint32_t tc = 0x4A1C3320;
  uint32_t rc = 0xABCD1234;
  uint32_t encAt = 0xCF412161;
  uint16_t tempOut[8] = {0};

  mifareCrypto1_ctx_t ctx = {0};


  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderInit(&ctx, key, uid, tc, rc, tempOut), "Didn't get ERR_OK to authReaderInit 1");

  uint16_t const expectedTempOut[8] = {0x805A, 0x0058, 0x007E, 0x8095, 0x0045, 0x0070, 0x00E8, 0x800E};
  TEST_ASSERT_EQUAL_HEX16_ARRAY_MESSAGE(expectedTempOut, tempOut, 8, "Didn't set Reader -> Tag reply correctly 1");
  TEST_ASSERT_EQUAL_INT_MESSAGE(ERR_OK, mifareCrypto1_authReaderHandleTagReply(&ctx, encAt), "Didn't get ERR_OK to authReaderHandleTagReply 1");

  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x6BAD4B30, ctx.ks[0], "ks0 not set correctly (1)");
  TEST_ASSERT_EQUAL_HEX32_MESSAGE(0x8FEB36B9, ctx.ks[1], "ks1 not set correctly (1)");
}




void test_mifare_crypto1_linear_feedback(void){
  //Test the Linear Feedback generator
  uint64_t inputVector[8] = {
    0x7631573AF53B, 0xACDFAC08454, 0x5B0DAD2A26AB, 0xBE2C8BA914E,
    0x6F0939F891F6, 0x3CEBD7C79C16, 0x6385489EB973, 0x4A0FFA116208
  };

  int expectedOutput[8] = {
    0, 1, 0, 0,
    1, 0, 1, 0,
  };

  int output[8] = {0};

  for(int i = 0; i < 8; i++){
    output[i] = mifareCrypto1_generateFbBitLinear(inputVector[i]);
  }

  TEST_ASSERT_EQUAL_INT_ARRAY_MESSAGE(expectedOutput, output, 8, "Linear feedback incorrect");
}

void test_mifare_crypto1_non_linear_feedback(void){
  //Test the Non-Linear Feedback generator
  uint64_t inputVector[8] = {
    0xFEE0FF819DA3, 0xFF707FC0CED1, 0xDFEE0FF819DA, 0xD37FB83FE067,
    0x67EA09413699, 0x1569BFDC1FF0, 0x58AB4DFEE0FF, 0x558AB4DFEE0F
  };

  int expectedOutput[8] = {
    1, 1, 1, 1,
    0, 0, 1, 0
  };

  int output[8] = {0};

  for(int i = 0; i < 8; i++){
    output[i] = mifareCrypto1_generateFbBitNonLinear(inputVector[i]);
  }

  TEST_ASSERT_EQUAL_INT_ARRAY_MESSAGE(expectedOutput, output, 8, "Non Linear feedback incorrect");
}


void test_mifare_crypto1_suc(void){
  uint32_t testVectorsIn[] = {
    0x009080A2,
    0x4A1C3320
  };

  uint32_t testVectors64Out[] = {
    0xB172DED3,
    0x64AE4245
  };

  uint32_t testVectors96Out[] = {
    0xCC1B98DE,
    0x90DE12A4
  };

  uint32_t out64[2] = {0};
  uint32_t out96[2] = {0};

  for(int i = 0; i < 2; i++){
    out64[i] = mifareCrypto1_suc(testVectorsIn[i], 64);
    out96[i] = mifareCrypto1_suc(testVectorsIn[i], 96);
  }

  TEST_ASSERT_EQUAL_HEX32_ARRAY_MESSAGE(testVectors64Out, out64, 2, "suc64 incorrect");
  TEST_ASSERT_EQUAL_HEX32_ARRAY_MESSAGE(testVectors96Out, out96, 2, "suc96 incorrect");
}

void test_mifare_crypto1_byteSwap32(void){
  uint32_t testVectorsIn[4] = {
    0x4A1C3320,
    0x267542A2,
    0x097B4825,
    0x8D4E7BCB
  };

  uint32_t testVectorsOut[4] = {
    0x5238CC04,
    0x64AE4245,
    0x90DE12A4,
    0xB172DED3
  };

  uint32_t out[4] = {0};
  for(int i = 0; i < 4; i++){
    out[i] = mifareCrypto1_byteSwap32(testVectorsIn[i]);
  }
  TEST_ASSERT_EQUAL_HEX32_ARRAY_MESSAGE(testVectorsOut, out, 4, "byteSwap32 error");
}

void test_mifare_crypto1_endianSwap48(void){
  uint64_t testVectorsIn[2] = {
    0xA0A1A2A3A4A5,
    0x204752454154
  };

  uint64_t testVectorsOut[2] = {
    0xA5A4A3A2A1A0,
    0x544145524720
  };

  uint64_t out[2] = {0};
  for(int i = 0; i < 2; i++){
    out[i] = mifareCrypto1_endianSwap48(testVectorsIn[i]);
  }
  TEST_ASSERT_EQUAL_HEX64_ARRAY_MESSAGE(testVectorsOut, out, 2, "endianSwap48 error");
}

void test_mifare_crypto1_round(void){
  //For now, only test FBTYPE_LINEAR, as that's the only one used
  uint64_t testVectorsLfsrStateIn[4] = {
    0x544145524720,
    0x9BFDC1FF033B,
    0x2AD37FB83FE0,
    0xB93A30544145
  };

  uint64_t testVectorsLfsrStateOut[4] = {
    0x2A20A2A92390,
    0x4DFEE0FF819D,
    0x1569BFDC1FF0,
    0x5C9D182A20A2
  };

  uint8_t testVectorsBitIn[4] = {
    1, 0, 0, 1
  };

  int testVectorsReturnOut[4] = {
    1, 0, 0, 0
  };

  int outReturn[4] = {0};
  uint64_t lfsrState[4] = {0};

  for(int i = 0; i < 4; i++){
    mifareCrypto1_ctx_t testCtx = {
      .lfsr = testVectorsLfsrStateIn[i]
    };

    outReturn[i] = mifareCrypto1_round(&testCtx, testVectorsBitIn[i], MIFARE_CRYPTO1_FBTYPE_LINEAR);
    lfsrState[i] = testCtx.lfsr;
  }

  TEST_ASSERT_EQUAL_INT_ARRAY_MESSAGE(testVectorsReturnOut, outReturn, 4, "Incorrect outReturn");
  TEST_ASSERT_EQUAL_HEX64_ARRAY_MESSAGE(testVectorsLfsrStateOut, lfsrState, 4, "Incorrect LFSR state");
}

void test_mifare_crypto1_calc_parity_bit(void){
  uint8_t testVectorsParityByteIn[4] = {
    0xD5, 0xB3, 0x48, 0x2C
  };

  uint64_t testVectorsLfsrStateIn[4] = {
    0x11752C4CEFA5,
    0x5811752C4CEF,
    0xC25811752C4C,
    0xBAC25811752C
  };

  int testVectorsReturnOut[4] = {
    1, 0, 0, 1
  };

  int outReturn[4] = {0};
  uint64_t lfsrState[4] = {0};

  for(int i = 0; i < 4; i++){
    mifareCrypto1_ctx_t testCtx = {
      .lfsr = testVectorsLfsrStateIn[i]
    };

    outReturn[i] = mifareCrypto1_calcParityBit(&testCtx, testVectorsParityByteIn[i]);
    lfsrState[i] = testCtx.lfsr;
  }

  TEST_ASSERT_EQUAL_INT_ARRAY_MESSAGE(testVectorsReturnOut, outReturn, 4, "Incorrect outReturn");
  TEST_ASSERT_EQUAL_HEX64_ARRAY_MESSAGE(testVectorsLfsrStateIn, lfsrState, 4, "Incorrect LFSR state");
}

void setUp(void){
}

void tearDown(void){
}



int main(void){
  //syslog_init(NULL, SYSLOG_LEVEL_VERBOSE, put_ch);
  UNITY_BEGIN();
  RUN_TEST(test_mifare_crypto1_linear_feedback);
  RUN_TEST(test_mifare_crypto1_non_linear_feedback);
  RUN_TEST(test_mifare_crypto1_byteSwap32);
  RUN_TEST(test_mifare_crypto1_endianSwap48);
  RUN_TEST(test_mifare_crypto1_suc);
  RUN_TEST(test_mifare_crypto1_round);
  RUN_TEST(test_mifare_crypto1_calc_parity_bit);
  RUN_TEST(test_mifare_crypto1_auth_sym_key_sym_rc);
  RUN_TEST(test_mifare_crypto1_auth_non_sym_key_sym_rc);
  RUN_TEST(test_mifare_crypto1_auth_non_sym_key_non_sym_rc);
  RUN_TEST(test_mifare_crypto1_auth_diff_uid);
  return UNITY_END();

}


