//Mifare Classic Driver
//(C) TechnicallyObsolete 2025
#include "tag_mifare_classic.h"
#include "rfid_generic.h"
#include "iso14443a.h"

#include "error_codes.h"
#include "utils/syslog.h"
#include "utils/bit_utils.h"

typedef enum {
  TAG_MIFARE_CLASSIC_MODE_NONE = 0,
  TAG_MIFARE_CLASSIC_MODE_CRC = 1,
  TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC = 3
} tagMifareClassic_transferMode_e;

static int tagMifareClassic_calcParity(uint8_t in);
static int tagMifareClassic_dataReformatOut(tagMifareClassic_ctx_t *ctx, uint8_t const *dataIn, uint16_t dataOut[18], uint_least8_t len, tagMifareClassic_transferMode_e transferMode);
static int tagMifareClassic_dataReformatIn(tagMifareClassic_ctx_t *ctx, uint16_t const *dataIn, uint8_t *dataOut, uint_least8_t len, tagMifareClassic_transferMode_e transferMode);
static int tagMifareClassic_discardAck(tagMifareClassic_ctx_t *ctx);

int tagMifareClassic_init(tagMifareClassic_ctx_t *ctx, void * readerCtx, rfidGeneric_fn_sendFrameWaitForReplyExtParity_t sendFrameWaitForReplyExtParity){
  if (!ctx) {
    return ERR_BAD_PARAM;
  }

  if (!sendFrameWaitForReplyExtParity){
    return ERR_BAD_PARAM;
  }

  ctx->readerCtx = readerCtx;
  ctx->sendFrameWaitForReplyExtParity = sendFrameWaitForReplyExtParity;

  return ERR_OK;
}


int tagMifareClassic_authBlock(tagMifareClassic_ctx_t *ctx, uint64_t key, tagMifareClassic_key_e keyAB, uint32_t readerChallenge, uint32_t uid, uint8_t blockAddr){

  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (!ctx->sendFrameWaitForReplyExtParity){
    return ERR_DATAERROR;
  }

  uint8_t authReq[4] = {TAG_MIFARE_CLASSIC_CMD_AUTH_A, blockAddr, 0x00, 0x00};

  if (keyAB == TAG_MIFARE_CLASSIC_KEY_B){
    authReq[0] = TAG_MIFARE_CLASSIC_CMD_AUTH_B;
  }

  tagMifareClassic_dataReformatOut(ctx, authReq, ctx->temp, 2, TAG_MIFARE_CLASSIC_MODE_CRC);

  //Crypto1 uses custom parity bit encoding,
  // so we have to request the RFID TRX to not add parity or CRC
  uint32_t flagWord = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY;
  uint32_t recvSts = 0;

  int sendSts = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, ctx->temp, 4, ctx->temp, TAG_MIFARE_CLASSIC_TEMP_BUF_SIZE, &recvSts);

  if (sendSts != 4){
    syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "Incorrect reply to AUTH, expected 4 words tag challenge but saw %d", sendSts);
    return ERR_DATAERROR;
  }

  uint32_t tagChallenge = 0;
  //Can't use the regular bitUtils bufferToU32, as temp is 16 bits (to allow for parity bits)
  //
  //Note, parity bits aren't checked here
  tagChallenge = ((ctx->temp[0] & 0xFF) << 24) | ((ctx->temp[1] & 0xFF) << 16) | ((ctx->temp[2] & 0xFF) << 8) | (ctx->temp[3] & 0xFF);
  syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "tagChallenge = 0x%08X", tagChallenge);


  uint16_t tempToTag[8] = {0};
  int returnVal = mifareCrypto1_authReaderInit(&(ctx->crypto1Ctx), key, uid, tagChallenge, readerChallenge, tempToTag);
  if (returnVal < 0){
    goto end;
  }
  //tempToTag has been encrypted and includes parity bits, so is OK to send directly
  sendSts = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, tempToTag, 8, ctx->temp, TAG_MIFARE_CLASSIC_TEMP_BUF_SIZE, &recvSts);
  if (sendSts != 4){
    syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "Incorrect reply to AUTH, expected 4 words tag reply but saw %d", sendSts);
    return ERR_DATAERROR;
  }

  uint32_t tagReply = ((ctx->temp[0] & 0xFF) << 24) | ((ctx->temp[1] & 0xFF) << 16) | ((ctx->temp[2] & 0xFF) << 8) | (ctx->temp[3] & 0xFF);

  returnVal = mifareCrypto1_authReaderHandleTagReply(&(ctx->crypto1Ctx), tagReply);
end:
  return returnVal;
}

int tagMifareClassic_blockRead(tagMifareClassic_ctx_t *ctx, uint8_t blockAddr, uint8_t out[16]){
  uint8_t readCmd[2] = {TAG_MIFARE_CLASSIC_CMD_READ, blockAddr};

  syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Reading block %d", blockAddr);
  int returnVal = tagMifareClassic_dataReformatOut(ctx, readCmd, ctx->temp, sizeof(readCmd), TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);
  if (returnVal < 0){
    goto end;
  }

  uint32_t flagWord = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY;
  uint32_t recvSts = 0;
  int sendSts = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, ctx->temp, 4, ctx->temp, TAG_MIFARE_CLASSIC_TEMP_BUF_SIZE, &recvSts);
  if (sendSts < 0){
    returnVal = sendSts;
    goto end;
  }

  returnVal = tagMifareClassic_dataReformatIn(ctx, ctx->temp, out, sendSts, TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);

  if (returnVal < 0){
    goto end;
  }

end:
  return returnVal;
}

int tagMifareClassic_safeBlockWrite(tagMifareClassic_ctx_t *ctx, uint8_t blockAddr, uint8_t const in[16]){
  if ((blockAddr & 3) == 3){
    //It's a key or access control byte, don't write it
    return ERR_NO_PRIV;
  }
  syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Writing block %d", blockAddr);

  uint8_t wrCmd[2] = {TAG_MIFARE_CLASSIC_CMD_WRITE, blockAddr};

  uint32_t flagWord = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY |  RFID_GENERIC_FLAG_NO_RETRY;
  uint32_t recvSts = 0;


  int returnVal = tagMifareClassic_dataReformatOut(ctx, wrCmd, ctx->temp, sizeof(wrCmd), TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);
  if (returnVal < 0){
    goto end;
  }

  //TODO:
  // the CR95HF driver doesn't properly handle partial (4 bit) ACK frames
  // so for now, ignore the return code
  returnVal = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, ctx->temp, 4, ctx->temp, 4, &recvSts);


  //Handle the ACK from the write command
  //for now, just advance the LFSR 4 bits
  // don't check the ACK for the moment
  tagMifareClassic_discardAck(ctx);


  //Now set up the write data
  tagMifareClassic_dataReformatOut(ctx, in, ctx->temp, 16, TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);

  returnVal = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, ctx->temp, 18, ctx->temp, 4, &recvSts);

  //Handle the ACK from the write data
  // again, not checking the ACK at the moment
  //
  tagMifareClassic_discardAck(ctx);

end:
  //TODO, add checks for ACK write status
  // this is also potentially going to fail without some delay time between write and next command
  return 0;
}

int tagMifareClassic_valueRead(tagMifareClassic_ctx_t *ctx, uint8_t blockAddr, uint32_t *value){
  if (!value){
    return ERR_BAD_PARAM;
  }

  uint8_t temp[16] = {0};

  int returnVal = tagMifareClassic_blockRead(ctx, blockAddr, temp);
  if (returnVal < 0){
    goto end;
  }

  //Check the value blocks
  for(int i = 0; i < 4; i++){
    //Bytes 0, 8 (and 1, 9 etc) should be equal
    if (temp[i] != temp[i + 8]){
      syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "Value block error, [%d] = 0x%02X but [%d] = 0x%02X", i, temp[i], i + 8, temp[i + 8]);
      returnVal = ERR_DATAERROR;
      goto end;
    }

    //Bytes 0, 4 should be inverse
    uint8_t tempInv = ~temp[i];
    if (tempInv != temp[i + 4]){
      syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "Value block error, ~[%d] = 0x%02X but [%d] = 0x%02X", i, tempInv, i + 4, temp[i + 4]);
    }
  }

  //If we got this far, then the value is OK
  //Not checking the addr field at the moment
  //
  //Is this little endian or big endian?
  // Datasheet says big endian, but PM3 updates blocks as little endian
  *value = bitUtils_bufferToUint32(temp, BIT_UTILS_LITTLE_ENDIAN);
  returnVal = ERR_OK;
end:
  return returnVal;
}

int tagMifareClassic_halt(tagMifareClassic_ctx_t *ctx, bool isAuthenticated){
  uint8_t haltCmd[2] = {TAG_MIFARE_CLASSIC_CMD_HALT, 0};

  syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Halt, isAuthenticated = %d", isAuthenticated);

  tagMifareClassic_transferMode_e mode = TAG_MIFARE_CLASSIC_MODE_CRC;
  if (isAuthenticated){
    mode = TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC;
  }

  int returnVal = tagMifareClassic_dataReformatOut(ctx, haltCmd, ctx->temp, sizeof(haltCmd), mode);
  if (returnVal < 0){
    goto end;
  }

  //As the device will HALT, don't expect a reply
  uint32_t flagWord = RFID_GENERIC_FLAG_NO_CRC_TX | RFID_GENERIC_FLAG_NO_CRC_RX | RFID_GENERIC_FLAG_NO_PARITY | RFID_GENERIC_FLAG_NO_REPLY;
  uint32_t recvSts = 0;
  returnVal = ctx->sendFrameWaitForReplyExtParity(ctx->readerCtx, flagWord, ctx->temp, 4, ctx->temp, TAG_MIFARE_CLASSIC_TEMP_BUF_SIZE, &recvSts);

  //Clamp to zero, since we're not interested in the data back from the tag
  if (returnVal >= 0){
    returnVal = 0;
  }

end:
  return returnVal;
}


//Helper functions

//For the moment, just discard the ACKs
//  This requires moving the LFSR forward 4 bits
int tagMifareClassic_discardAck(tagMifareClassic_ctx_t *ctx){
  int returnVal = ERR_OK;

  for(int i = 0; i < 4; i++){
    returnVal = mifareCrypto1_round(&ctx->crypto1Ctx, 0, MIFARE_CRYPTO1_FBTYPE_LINEAR);
    if (returnVal < 0){
      break;
    }
  }

  return returnVal;
}

static int tagMifareClassic_calcParity(uint8_t in){
  int sum = 1;
  for(int i = 0; i < 8; i++){
    if (in & (1 << i)){
      sum ^= 1;
    }
  }
  return sum;
}

//General purpose helper function, to convert from u8 to u16 and add parity (optionally CRC and encryption)
static int tagMifareClassic_dataReformatOut(tagMifareClassic_ctx_t *ctx, uint8_t const *dataIn, uint16_t dataOut[18], uint_least8_t len, tagMifareClassic_transferMode_e transferMode){
  bool encrypt = (transferMode == TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);
  bool appendCrc = (transferMode == TAG_MIFARE_CLASSIC_MODE_CRC) || (transferMode == TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);

  if ((len > 18)|| ((len > 16) && appendCrc)){
    return ERR_MSG_OVERLENGTH;
  }

  //Calculate the plaintext parity bits first
  for(uint_least8_t i = 0; i < len; i++){
    dataOut[i] = tagMifareClassic_calcParity(dataIn[i]) << 15;
  }


  //Now encrypt the data
  //This function preserves the upper 8 bits but writes the lower 16b
  for(uint_least8_t i = 0; i < len; i++){
    if(encrypt){
      mifareCrypto1_cryptByteOut16b(&(ctx->crypto1Ctx), dataIn[i], &dataOut[i], MIFARE_CRYPTO1_FBTYPE_LINEAR);
    } else {
      dataOut[i] |= dataIn[i];
    }
  }

  syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Data to TX:");
  for(uint_least8_t i = 0; i < len; i++){
    uint16_t tempParity = tagMifareClassic_calcParity(dataOut[i]);

    bool parityErr = (tempParity != (dataOut[i] & 0x8000));
    syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Data in: 0x%02X Out: 0x%04X Parity out: %d err: %d", dataIn[i], dataOut[i], tempParity, parityErr);
  }

  if (appendCrc){
    uint16_t crc = iso14443a_crcA(dataIn, len);
    uint8_t crcMsb = crc >> 8;
    uint8_t crcLsb = crc & 0xFF;

    dataOut[len] = tagMifareClassic_calcParity(crcLsb) << 15;
    dataOut[len + 1] = tagMifareClassic_calcParity(crcMsb) << 15;


    if (encrypt){
      mifareCrypto1_cryptByteOut16b(&(ctx->crypto1Ctx), crcLsb, &dataOut[len], MIFARE_CRYPTO1_FBTYPE_LINEAR);
      mifareCrypto1_cryptByteOut16b(&(ctx->crypto1Ctx), crcMsb, &dataOut[len + 1], MIFARE_CRYPTO1_FBTYPE_LINEAR);

      syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "CRC 0x%02X 0x%02X out 0x%04X 0x%04X", crcLsb, crcMsb, dataOut[len], dataOut[len + 1]);
    } else {
      dataOut[len] |= crcLsb;
      dataOut[len + 1] |= crcMsb;
    }

    return len + 2;
  } else {
    return len;
  }
}

static int tagMifareClassic_dataReformatIn(tagMifareClassic_ctx_t *ctx, uint16_t const *dataIn, uint8_t *dataOut, uint_least8_t len, tagMifareClassic_transferMode_e transferMode){
  bool decrypt = (transferMode == TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);
  bool checkCrc = (transferMode == TAG_MIFARE_CLASSIC_MODE_CRC) || (transferMode == TAG_MIFARE_CLASSIC_MODE_CRYPT_CRC);

  if ((len == 0) || ((len < 2) && checkCrc)){
    return ERR_MSG_UNDERLENGTH;
  }

  if (len > 48){
    //No replies from the card should be this long, so something failed
    syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "Too many bytes (%d) to reformatIn", len);
    return ERR_MSG_OVERLENGTH;
  }


  //On the RX side, don't bother checking the parity bits for now
  uint16_t crcRecv = 0;

  for(uint_least8_t i = 0; i < len; i++){
    uint16_t temp = 0;
    //uint8_t decParityBit = mifareCrypto1_calcParityBit(&ctx->crypto1Ctx, dataIn[i] & 0xFF);
    if(decrypt){
      mifareCrypto1_cryptByteOut16b(&(ctx->crypto1Ctx), dataIn[i] & 0xFF, &temp, MIFARE_CRYPTO1_FBTYPE_LINEAR);

      //The input data stream will have a certain number of bytes, and regardless of whether the CRC is checked or not,
      // the LFSR still needs to be advanced by that many bytes

      if (checkCrc && (i == (len - 2))){
        crcRecv = temp & 0xFF;
      } else if (checkCrc && (i == (len - 1))){
        crcRecv |= ((temp & 0xFF) << 8);
      } else {
        dataOut[i] = temp & 0xFF;
      }
     
    } else {
      dataOut[i] = dataIn[i];
    }
    syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Data in: 0x%04X inp %d Out: 0x%04X outp: %d", dataIn[i], tagMifareClassic_calcParity(dataIn[i] & 0xFF), temp, tagMifareClassic_calcParity(temp & 0xFF));
  }

  if (checkCrc){
    uint16_t crcExp = iso14443a_crcA(dataOut, len - 2);

    if (crcExp != crcRecv){
      syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_ERROR, "CRC recv: 0x%04X exp: 0x%04X", crcRecv, crcExp);
      return ERR_BAD_CHECKSUM;
    } else {
      syslog_logFnName(TAG_MIFARE_CLASSIC_LOG_CTX, SYSLOG_LEVEL_DEBUG, "CRC OK, 0x%04X", crcRecv);
    }
  }

  return ERR_OK;
}
