//Mifare Classic Driver
//(C) TechnicallyObsolete 2019, 2025
#include "mifare_crypto1.h"
#include <stdbool.h>
#include "error_codes.h"

#include "utils/bit_utils.h"
#include "utils/syslog.h"

#include <stdio.h>

//Bit ordering is strange here
//bits within the byte are reversed, but not the overall bytes
uint32_t mifareCrypto1_byteSwap32(uint32_t in){
  uint8_t temp[4] = {0};

  bitUtils_uint32ToBuffer(in, temp, BIT_UTILS_BIG_ENDIAN);

  for(int i = 0; i < 4; i++){
    temp[i] = bitUtils_rev8(temp[i]);
  }

  in = bitUtils_bufferToUint32(temp, BIT_UTILS_BIG_ENDIAN);
  return in;
}


//Used to swap the bytes in the key to the correct order
uint64_t mifareCrypto1_endianSwap48(uint64_t in){
  uint8_t temp[8] = {0};

  bitUtils_uint64ToBuffer(in, temp, BIT_UTILS_BIG_ENDIAN);
  in = bitUtils_bufferToUint64(temp, BIT_UTILS_LITTLE_ENDIAN);

  in >>= 16;
  return in;
}

#define MIFARE_CRYPTO1_NO_LUT

//Inputs to this:
// in bit 0 = A (y0 e.g. 9)
//        1 = B (y1 e.g. 11))
//        2 = C (y2 e.g. 13)
//        3 = D (y3 e.g. 15)
int mifareCrypto1_fb(uint_least8_t in){
#ifdef MIFARE_CRYPTO1_NO_LUT
  //Notes:
  // x∧y is AND
  // x∨y is OR
  bool y0 = (in >> 0) & 1;
  bool y1 = (in >> 1) & 1;
  bool y2 = (in >> 2) & 1;
  bool y3 = (in >> 3) & 1;

  return ((y0 | y1) ^ (y0 & y3)) ^ (y2 & ((y0 ^ y1) | y3));
#else
  in &= 0xF;
  uint16_t filter_fb = 0xB48E;

  if (filter_fb & (1 << in)){
    return 1;
  } else{
    return 0;
  }
#endif
}

static int mifareCrypto1_fa(uint_least8_t in){
#ifdef MIFARE_CRYPTO1_NO_LUT
  bool y0 = (in >> 0) & 1;
  bool y1 = (in >> 1) & 1;
  bool y2 = (in >> 2) & 1;
  bool y3 = (in >> 3) & 1;
return ((y0 & y1) | y2)  ^  ((y0 ^ y1) & (y2 | y3));
#else
in &= 0xF;
uint16_t filter_fa = 0x9E98;
if (filter_fa & (1 << in)){
  return 1;
} else{
  return 0;
}
#endif
}

static bool mifareCrypto1_fc(uint_least8_t in){
#ifdef MIFARE_CRYPTO1_NO_LUT
  bool y0 = (in & 1);
  bool y1 = (in & 2) >> 1;
  bool y2 = (in & 4) >> 2;
  bool y3 = (in & 8) >> 3;
  bool y4 = (in & 16) >> 4;

  return (y0 | ((y1 | y4) & (y3 ^ y4))) ^ ((y0 ^ (y1 & y3))& ((y2 ^ y3) | (y1& y4)));
#else
  in &= 0x1F;
  uint32_t filter_fc = 0xEC57E80A;

  if (filter_fc & (1 << in)){
    return true;
  } else{
    return false;
  }
#endif
}


//Non-linear feedback
int mifareCrypto1_generateFbBitNonLinear(uint64_t in){
  uint_least8_t inputFb1 = (((in >> 9) & 1) << 0) | (((in >> 11) & 1) << 1) | (((in >> 13) & 1) << 2) | (((in >> 15) & 1) << 3);
  uint_least8_t inputFa2 = (((in >> 17) & 1) << 0) | (((in >> 19) & 1) << 1) | (((in >> 21) & 1) << 2) | (((in >> 23) & 1) << 3);
  uint_least8_t inputFa3 = (((in >> 25) & 1) << 0) | (((in >> 27) & 1) << 1) | (((in >> 29) & 1) << 2) | (((in >> 31) & 1) << 3);
  uint_least8_t inputFb4 = (((in >> 33) & 1) << 0) | (((in >> 35) & 1) << 1) | (((in >> 37) & 1) << 2) | (((in >> 39) & 1) << 3);
  uint_least8_t inputFa5 = (((in >> 41) & 1) << 0) | (((in >> 43) & 1) << 1) | (((in >> 45) & 1) << 2) | (((in >> 47) & 1) << 3);

  uint_least8_t inputFc = 0;
  inputFc |= mifareCrypto1_fb(inputFb1) << 0;
  inputFc |= mifareCrypto1_fa(inputFa2) << 1;
  inputFc |= mifareCrypto1_fa(inputFa3) << 2;
  inputFc |= mifareCrypto1_fb(inputFb4) << 3;
  inputFc |= mifareCrypto1_fa(inputFa5) << 4;

  int returnVal = mifareCrypto1_fc(inputFc) ? 1 : 0;

  return returnVal;
}

//Linear feedback
int mifareCrypto1_generateFbBitLinear(uint64_t in){
  uint64_t newBit =
          (in >> 43) ^ (in >> 42) ^ (in >> 41) ^ (in >> 39) ^
          (in >> 35) ^ (in >> 29) ^ (in >> 27) ^ (in >> 25) ^
          (in >> 24) ^ (in >> 19) ^ (in >> 17) ^ (in >> 15) ^
          (in >> 14) ^ (in >> 12) ^ (in >> 10) ^ (in >> 9) ^
          (in >> 5) ^ (in >> 0);

  newBit &= 1;
  return newBit;
}



//Generate successive LFSR outputs
//
//this is referred to as 'suc' in the research papers
// e.g. suc96(value)
//
//In this case: mifareCrypto1_suc(value, 96);
uint32_t mifareCrypto1_suc(uint32_t in, uint_least16_t steps){

  in = mifareCrypto1_byteSwap32(in);

  for(uint_least16_t i = 0; i < steps; i++){
    uint32_t nextBit = (((in >> 15) & 1) ^ ((in >> 13) & 1) ^ ((in >> 12) ^ 1) ^ ((in >> 10) ^ 1)) & 1;
    in = (in << 1) | nextBit;
  }

  in = mifareCrypto1_byteSwap32(in);

  return in;
}

//Encrypt a single bit, with selectable feedback
int mifareCrypto1_round(mifareCrypto1_ctx_t *ctx, int in, mifareCrypto1_fbtype_e fb){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  uint64_t temp = ctx->lfsr;

  int lf = mifareCrypto1_generateFbBitLinear(temp);
  int nf = mifareCrypto1_generateFbBitNonLinear(temp);

  uint64_t newBit = in & 1;
  if ((fb == MIFARE_CRYPTO1_FBTYPE_LINEAR) || (fb == MIFARE_CRYPTO1_FBTYPE_BOTH)){
    newBit ^= lf;
  }

  if((fb == MIFARE_CRYPTO1_FBTYPE_NONLINEAR) || (fb == MIFARE_CRYPTO1_FBTYPE_BOTH)){
    newBit ^= nf;
  }

  temp = (temp >> 1) ^ (newBit << 47ULL);

  ctx->lfsr = temp;
  return nf;
}


//Generate a parity bit, but don't advance the state of the LFSR
int mifareCrypto1_calcParityBit(mifareCrypto1_ctx_t *ctx, uint8_t in){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  //Step 1: Calculate odd parity
  uint8_t parity = 1;
  for(int i = 0; i < 8; i++){
    if (in & (1 << i)){
      parity ^= 1;
    }
  }

  //Step 2: Encrypt the parity bit
  int nf = mifareCrypto1_generateFbBitNonLinear(ctx->lfsr);

  int returnVal = (parity ^ nf) & 1;
  return returnVal;
}

//Generate 8 bits of keystream
int mifareCrypto1_updateKeystreamByte(mifareCrypto1_ctx_t *ctx, uint8_t byteIn, uint8_t *byteOut, mifareCrypto1_fbtype_e fb){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  uint32_t out = 0;
  uint32_t bitIn = 0;

  for(int i = 0; i < 8; i++){
    bitIn = (byteIn>> i) & 1;
    out |= mifareCrypto1_round(ctx, bitIn, fb) << i;
  }

  if (byteOut){
    *byteOut = out;
  }
  return ERR_OK;
}

//Generate 32 bits of keystream
int mifareCrypto1_updateKeystreamWord(mifareCrypto1_ctx_t *ctx, uint32_t wordIn, uint32_t *wordOut, mifareCrypto1_fbtype_e fb){
  int returnVal = ERR_OK;
  uint8_t temp[4] = {0};

  bitUtils_uint32ToBuffer(wordIn, temp, BIT_UTILS_BIG_ENDIAN);

  for(int i = 0; i < 4; i++){
    returnVal = mifareCrypto1_updateKeystreamByte(ctx, temp[i], &temp[i], fb);
    if (returnVal < 0){
      goto end;
    }
  }

  if (wordOut){
    *wordOut = bitUtils_bufferToUint32(temp, BIT_UTILS_BIG_ENDIAN);
  }

end:
  return returnVal;
}

//Encrypt or decrypt a byte, with separate parity bit
//If the parity bit is ste, then *parityOut = 0x80, otherwise 0x00
int mifareCrypto1_cryptByte(mifareCrypto1_ctx_t *ctx, uint8_t byteIn, uint8_t *byteOut, uint8_t *parityOut, mifareCrypto1_fbtype_e fb){
  uint8_t keyStream = 0;
  int returnVal = mifareCrypto1_updateKeystreamByte(ctx, 0, &keyStream, fb);

  if (byteOut){
    *byteOut = byteIn ^ keyStream;
  }

  if (parityOut){
    *parityOut = mifareCrypto1_calcParityBit(ctx, byteIn) << 7;
  }
  return returnVal;
}

//Output into lower part of uint16_t, parity in the upper part
//Parity is in bit 15
int mifareCrypto1_cryptByteOut16b(mifareCrypto1_ctx_t *ctx, uint8_t byteIn, uint16_t *byteOut, mifareCrypto1_fbtype_e fb){
  uint8_t tempData = 0;
  uint8_t tempParity = 0;

  int returnVal = mifareCrypto1_cryptByte(ctx, byteIn, &tempData, &tempParity, fb);

  if (byteOut){
    *byteOut = (tempParity << 8) | (tempData);

    syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "In: 0x%02X Out: 0x%04X Parity: %d", byteIn, *byteOut, tempParity >> 7);
  }
  return returnVal;
}


int mifareCrypto1_cryptWord(mifareCrypto1_ctx_t *ctx, uint32_t wordIn, uint32_t *wordOut, mifareCrypto1_fbtype_e fb){
  uint32_t keyStream = 0ULL;
  int returnVal = mifareCrypto1_updateKeystreamWord(ctx, 0, &keyStream, fb);

  if (wordOut){
    *wordOut = wordIn ^ keyStream;
  }
  return returnVal;
}

//Implements the first part of the auth process
//Accepts the tag challenge and generates the corresponding encrypted reader challenge and encrypted reader answer
int mifareCrypto1_authReaderInit(mifareCrypto1_ctx_t *ctx, uint64_t key, uint32_t uid, uint32_t tagChallenge, uint32_t readerChallenge, uint16_t dataOut[8]){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (!dataOut){
    return ERR_BAD_PARAM;
  }

  ctx->lfsr = mifareCrypto1_endianSwap48(key);
  ctx->tagChallenge = tagChallenge;

  //Step 1: calculate keystream values ks0 and ks1

  //ks0 is easy, because it's calculated on the UID, and as it's not sent to the tag we don't need to calculate parity
  mifareCrypto1_updateKeystreamWord(ctx, uid ^ tagChallenge, &(ctx->ks[0]), MIFARE_CRYPTO1_FBTYPE_LINEAR);

  //ks1 is more difficult, because we use that to encrypt the reader challenge, which is sent to the tag (so needs parity calculated)
  uint8_t readerChallengeByte[4] = {0};
  uint8_t encReaderChallengeByte[4] = {0};
  uint8_t encReaderChallengeParity[4] = {0};

  readerChallenge = mifareCrypto1_byteSwap32(readerChallenge);
  bitUtils_uint32ToBuffer(readerChallenge, readerChallengeByte, BIT_UTILS_BIG_ENDIAN);


  uint8_t ks1Byte[4] = {0};

  //The following could probably be replaced with calls to mifareCrypto1_cryptByteOut16b,
  // but for unit testing it's very convenient to calculate ks1
  //
  // So, duplicate (somewhat) the cryptOutByte in a way that allow preserving the ks1 value
  for(int i = 0; i < 4; i++){
    mifareCrypto1_updateKeystreamByte(ctx, readerChallengeByte[i], &ks1Byte[i], MIFARE_CRYPTO1_FBTYPE_LINEAR);
    encReaderChallengeByte[i] = readerChallengeByte[i] ^ ks1Byte[i];

    //Parity bit encoding uses bit 7 of byte, which is bit 15 of word
    encReaderChallengeParity[i] = mifareCrypto1_calcParityBit(ctx, readerChallengeByte[i]) << 7;

    syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "RC: 0x%02X eRC: 0x%02X Parity: %d",
      readerChallengeByte[i], encReaderChallengeByte[i], encReaderChallengeParity[i] >> 7);

    dataOut[i] = (encReaderChallengeParity[i] << 8) | (encReaderChallengeByte[i]);
  }

  //By this point, dataOut[0] to dataOut[3] = has the encrypted reader challenge

  //Store ks1 for the unit tests (not needed outside this function otherwise)
  ctx->ks[1] = bitUtils_bufferToUint32(ks1Byte, BIT_UTILS_BIG_ENDIAN);

  //encReaderChallenge is only used for debug print,
  // as the data that's being sent to the tag has already been encrypted and stored into encReaderChallengeByte
  uint32_t encReaderChallenge = readerChallenge ^ ctx->ks[1];

  uint32_t readerReply = mifareCrypto1_suc(tagChallenge, 64);

  uint8_t readerReplyByte[4] = {0};
  bitUtils_uint32ToBuffer(readerReply, readerReplyByte, BIT_UTILS_BIG_ENDIAN);

  //Encrypt the tag answer
  uint8_t encReaderReplyByte[4] = {0};
  uint8_t encReaderReplyParity[4] = {0};

  for(int i = 0; i < 4; i++){
    mifareCrypto1_cryptByte(ctx, readerReplyByte[i], &encReaderReplyByte[i], &encReaderReplyParity[i], MIFARE_CRYPTO1_FBTYPE_LINEAR);
    dataOut[i + 4] = (encReaderReplyParity[i] << 8) | (encReaderReplyByte[i]);
  }

  //Not strictly needed, but very convenient for debug
  uint32_t encReaderReply = bitUtils_bufferToUint32(encReaderReplyByte, BIT_UTILS_BIG_ENDIAN);

  //Debug printfs
  syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "UID=0x%08X TagChallenge=0x%08X ReaderChallenge=0x%08X", uid, tagChallenge, readerChallenge);
  syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Ks[0] = 0x%08X, Ks[1] = 0x%08X", ctx->ks[0], ctx->ks[1]);
  syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "encReaderChallenge = 0x%08X, readerReply = 0x%08X, encReaderReply = 0x%08X", encReaderChallenge, readerReply, encReaderReply);
  return ERR_OK;
}

//Handles the second part of the auth
int mifareCrypto1_authReaderHandleTagReply(mifareCrypto1_ctx_t *ctx, uint32_t tagReply){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  int returnVal = ERR_OK;

  uint32_t decAnswerTag = 0;
  returnVal = mifareCrypto1_cryptWord(ctx, tagReply, &decAnswerTag, MIFARE_CRYPTO1_FBTYPE_LINEAR);

  if (returnVal < 0){
    goto end;
  }

  uint32_t expectedAnswerTag = mifareCrypto1_suc(ctx->tagChallenge, 96);

  if (decAnswerTag == expectedAnswerTag){
    syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_INFO, "Auth OK: answerTag = 0x%08X decAnswerTag = 0x%08X matched", tagReply, decAnswerTag);
    returnVal = ERR_OK;
  } else {
    syslog_logFnName(MIFARE_CRYPTO1_LOG_CTX, SYSLOG_LEVEL_ERROR, "Auth Failed: answerTag = 0x%08X decAnswerTag = 0x%08X expected 0x%08X", tagReply, decAnswerTag, expectedAnswerTag);
    returnVal = ERR_DATAERROR;
  }

end:
  return returnVal;
}

