//CR95HF RFID driver
//(C) TechnicallyObsolete 2019, 2021, 2025
#include <string.h> //for memcmp
#include "cr95hf.h"
#include "error_codes.h"
#include "utils/xprintf.h"
#include "utils/syslog.h"

int cr95hf_waitUntilReady(cr95hf_ctx_t *ctx, uint8_t mask, int maxTries){
  int returnVal = ERR_OK;

  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (ctx->isUart){
    return ERR_OK;
  }

  if (ctx->defaultRetryCount < 0){
    return ERR_BAD_PARAM;
  }

  uint8_t pollCmd = CR95HF_SPI_POLL;
  uint8_t response = 0x00;

  int try = 0;
  while(!(response & mask)){
    cr95hf_ll_csLow(ctx);
    cr95hf_ll_tx(ctx, &pollCmd, 1);
    cr95hf_ll_rx(ctx, &response, 1);
    cr95hf_ll_csHigh(ctx);

    if (try > maxTries){
      returnVal = ERR_TMEOUT_NO_ACK;
      break;
    }
    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "Response = 0x%02X", response);
    if (maxTries > 0){
      try++;
    }
  }


  return returnVal;
}

int cr95hf_sendCmdNoWaitReply(cr95hf_ctx_t *ctx, uint8_t cmd, uint8_t const *data, uint8_t dataLen, bool appendFlags, uint8_t flags){
  cr95hf_ll_csLow(ctx);

  uint8_t header[3] = {CR95HF_SPI_SEND, cmd, dataLen};

  if (appendFlags){
    //Increase the data length reported to the CR95HF, as we'll be appending one extra byte
    header[2]++;
  }

  if (ctx->isUart){
    //Skip the SPI control byte in UART mode
    cr95hf_ll_tx(ctx, header + 1, 2);
  } else{
    cr95hf_ll_tx(ctx, header, 3);
  }

  cr95hf_ll_tx(ctx, data, dataLen);
  if (appendFlags){
    cr95hf_ll_tx(ctx, &flags, 1);
  }
  cr95hf_ll_csHigh(ctx);

  return ERR_OK;
}

int cr95hf_getReply(cr95hf_ctx_t *ctx, uint8_t *replyBuf, uint_least16_t maxReplyLen, uint8_t *returnCode){
  int returnVal = ERR_OK;

  uint8_t getCmd = CR95HF_SPI_READ;
  uint8_t replyTemp[2] = {0, 0};

  cr95hf_ll_csLow(ctx);
  if (!ctx->isUart){
    cr95hf_ll_tx(ctx, &getCmd, 1);
  }

  if (cr95hf_ll_rx(ctx, replyTemp, 2) != 2){
    //Bad length recv
    returnVal = ERR_HARDWARE_FAULT;
    goto end;
  };

  *returnCode = replyTemp[0];

  //Bits 9 and 8 of replyLen are stored in bits 6 and 5 of the result code
  uint_least16_t replyLen = replyTemp[1] | ((replyTemp[0] & 0x60) << 3);
  if (replyLen > maxReplyLen){
    //Read what bytes we can into the buffer
    //Note: the buffer will be corrupted, but this should prevent the SPI interface locking up
    int bytesLeftToClear = replyLen;
    while(bytesLeftToClear > maxReplyLen){
      cr95hf_ll_rx(ctx, replyBuf, maxReplyLen);
      bytesLeftToClear -= maxReplyLen;
    }

    if (bytesLeftToClear > 0){
      cr95hf_ll_rx(ctx, replyBuf, bytesLeftToClear);
    }

    returnVal = ERR_MSG_OVERLENGTH;
    goto end;
  }

  if (cr95hf_ll_rx(ctx, replyBuf, replyLen) != replyLen){
    returnVal = ERR_HARDWARE_FAULT;
  } else{
    returnVal = replyLen;
  }

end:
  cr95hf_ll_csHigh(ctx);
  return returnVal;
}

//Note: the last byte sent to this is a flags code, the last 3 bytes recv from this are errors/sig bits, collision idx (byte), collision idx (bit))
int cr95hf_sendCmdWaitForReply(cr95hf_ctx_t *ctx, uint8_t cmd, uint8_t const *data, uint_least16_t dataLen, uint8_t *replyBuf, uint_least16_t maxReplyLen, uint8_t *returnCode, bool appendFlags, uint8_t flags){
  syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_DEBUG, "Sending cmd=0x%02X Len: %d Append: %d (0x%02X)", cmd, dataLen, appendFlags, flags);
  int waitSts = cr95hf_waitUntilReady(ctx, CR95HF_FLAG_CMD_READY, ctx->defaultRetryCount);
  if (waitSts < 0){
    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_WARNING, "Error %d waiting for ready", waitSts);
    goto end;
  }

  waitSts = cr95hf_sendCmdNoWaitReply(ctx, cmd, data, dataLen, appendFlags, flags);
  if (waitSts < 0){
    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_WARNING, "Error %d sending command", waitSts);
    goto end;
  } else {
    xprintf("--> TX %d bytes:", dataLen);
    for(unsigned int i = 0; i < dataLen; i++){
      xprintf("%02X ", data[i]);
    }
    xprintf("\r\n");
  }

  waitSts = cr95hf_waitUntilReady(ctx, CR95HF_FLAG_DATA_PENDING, ctx->defaultRetryCount);
  if (waitSts < 0){
    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_WARNING, "Error %d getting reply", waitSts);
    goto end;
  }

  waitSts = cr95hf_getReply(ctx, replyBuf, maxReplyLen, returnCode);
  if (waitSts >= 0){
    xprintf("<-- RX %d bytes:", waitSts);
    for(int i = 0; i < waitSts; i++){
      xprintf("%02X ", replyBuf[i]);
    }
    xprintf("\r\n");
  }
end:
  return waitSts;
}

int cr95hf_sendCmdWaitForReplyRetry(cr95hf_ctx_t *ctx, uint8_t cmd, uint8_t const *data, uint_least16_t dataLen, uint8_t *replyBuf, uint_least16_t maxReplyLen, uint8_t *returnCode, unsigned int retry, bool appendFlags, uint8_t flags){

  int sendSts = ERR_OK;
  for(unsigned int i = 0; i < retry; i++){
    sendSts = cr95hf_sendCmdWaitForReply(ctx, cmd, data, dataLen, replyBuf, maxReplyLen, returnCode, appendFlags, flags);
    if (((sendSts == ERR_MSG_OVERLENGTH) || (sendSts >= 0)) && (*returnCode != CR95HF_RESULT_E_FRAME_WAIT_T_OUT)){
      return sendSts;
    }
  }
  return sendSts;
}

int cr95hf_init(cr95hf_ctx_t *ctx, rfidGeneric_protocol_e protocol){
  int returnVal = ERR_OK;

  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (ctx->defaultRetryCount < 0){
    ctx->defaultRetryCount = CR95HF_DEFAULT_RETRY_COUNT;
  }

  returnVal = cr95hf_ll_init(ctx);
  if (returnVal < 0){
    goto end;
  }

  returnVal = cr95hf_checkId(ctx);
  if (returnVal < 0){
    goto end;
  }
  //For the moment, hard code these timer values
  
  if (!ctx->customTiming){
    ctx->pp = 0x04;
    ctx->mm = 0x20;
    ctx->dd = 0x20;
  }

  if (protocol != RFID_GENERIC_PROTO_OFF){
    ctx->protocol = protocol;
  }

  //MIFARE classic seems to need this to be 1, tag with EEPROM needs 5

  uint8_t rfu1 = 0;
  uint8_t rfu2 = 0;
  uint8_t result = 0;
  //  //Ignore optional bytes
  // uint8_t params[2] = {0x02, 0x00}; //106kbps Tx, Rx
  if (protocol == RFID_GENERIC_PROTO_OFF){
    uint8_t paramsOff[2] = {0x00, 0x00};
    returnVal = cr95hf_sendCmdWaitForReply(ctx, CR95HF_CMD_PROTOCOL_SEL, paramsOff, sizeof (paramsOff), paramsOff, sizeof (paramsOff), &result, 0, 0);
  } else if (protocol == RFID_GENERIC_PROTO_ISO14443A){
    uint8_t params[7] = {0x02, 0x00, ctx->pp, ctx->mm, ctx->dd, rfu1, rfu2};
    returnVal = cr95hf_sendCmdWaitForReply(ctx, CR95HF_CMD_PROTOCOL_SEL, params, sizeof (params), params, sizeof (params), &result, 0, 0);

    if (returnVal < 0){
      goto end;
    }

    //Apply datasheet patch to improve TimerW in ISO14443A mode
    uint8_t timerAPatch[5] = {0x04, 0x3A, 0x00, 0x58, 0x04};
    returnVal = cr95hf_sendCmdWaitForReply(ctx, CR95HF_CMD_REG_WR, timerAPatch, sizeof(timerAPatch), timerAPatch, sizeof(timerAPatch), &result, 0, 0);


  } else {
    returnVal = ERR_BAD_PARAM;
  }
end:
  return returnVal;
}

int cr95hf_getFnPtr(rfidGeneric_fnPtr_t *ptr){
  if (!ptr){
    return ERR_BAD_PARAM;
  }

  ptr->init = (rfidGeneric_fn_init_t) cr95hf_init;

//  ptr->sendShortFrameWaitForReplyRetry = cr95hf_ptrSendShortFrame;
//  ptr->sendFrameWaitForReplyRetry = cr95hf_ptrSendRecv;
  return ERR_OK;
}

int cr95hf_checkId(cr95hf_ctx_t *ctx){
  int returnVal = ERR_OK;

  uint8_t cmdBuf[16] = {0};
  uint8_t resultCode = 0;

  int dataLen = cr95hf_sendCmdWaitForReply(ctx, CR95HF_CMD_IDN, cmdBuf, 0, cmdBuf, sizeof(cmdBuf), &resultCode, 0, 0);

  if (dataLen < 0){
    returnVal = dataLen;
    goto end;
  }

  if (dataLen < 4){
    returnVal = ERR_DATAERROR;
    goto end;
  }

  if (memcmp("NFC ", cmdBuf, 4) != 0){
    returnVal = ERR_DATAERROR;
    goto end;
  }

end:
  return returnVal;
}

int cr95hf_rfSendEof(cr95hf_ctx_t *ctx, uint8_t *recvBuf, uint_least16_t recvBufLen, int32_t *collisionPos);

int cr95hf_rfSendFrameWaitForReply(cr95hf_ctx_t *ctx, uint32_t flags, uint8_t const * sendBuf, uint_least16_t sendBufLen, uint8_t *recvBuf, uint_least16_t recvBufLen, uint32_t *recvSts){
  int returnVal = ERR_OK;

  uint8_t flagByte = 0;
  if (flags & RFID_GENERIC_FLAG_SPLIT_MASK){
    flagByte |= (flags & RFID_GENERIC_FLAG_SPLIT_MASK);
  } else {
    flagByte |= 8; //8 bits present
  }

  if (flags & RFID_GENERIC_FLAG_SPLIT_RX){
    flagByte |= CR95HF_TXFLAG_SPLIT_FRAME;
  }

  //CR95HF Doesn't support CRC on one side only
  bool noTxCrc = !!(flags & RFID_GENERIC_FLAG_NO_CRC_TX);
  bool noRxCrc = !!(flags & RFID_GENERIC_FLAG_NO_CRC_RX);

  if (noTxCrc != noRxCrc){
    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_ERROR, "NO_CRC_RX != NO_CRC_TX not supported, flags = 0x%08X RX: %d TX: %d", flags, noRxCrc, noTxCrc);
    return ERR_UNSUPPORTED_CMD;
  }

  if (!noTxCrc){
    flagByte |= CR95HF_TXFLAG_APPEND_CRC;
  }

  if (flags & RFID_GENERIC_FLAG_NO_PARITY){
    flagByte |= CR95HF_TXFLAG_PARITY_FRAME;
  }

  uint8_t returnCode = 0;
  bool appendFlags = false;

  if (ctx->protocol == RFID_GENERIC_PROTO_ISO14443A){
    appendFlags = true;
  }


  int retryCount = 3;
  if (flags & RFID_GENERIC_FLAG_NO_RETRY){
    retryCount = 1;
  }

  int replyLen = cr95hf_sendCmdWaitForReplyRetry(ctx, CR95HF_CMD_SEND_RECV, sendBuf, sendBufLen, recvBuf, recvBufLen, &returnCode, retryCount, appendFlags, flagByte);

  if (replyLen < 0){
    returnVal = replyLen;
  } else if (ctx->protocol == RFID_GENERIC_PROTO_ISO18092){
    //there's one appended data byte in ISO18092 byte, CRC status only
    if (replyLen < 1){
      returnVal = ERR_MSG_UNDERLENGTH;
    } else {
      returnVal = replyLen - 1;
    }
  } else if ((ctx->protocol == RFID_GENERIC_PROTO_ISO14443A) && (!noRxCrc)){
    //Remove the CRC as well
    if (replyLen < 5){
      returnVal = ERR_MSG_UNDERLENGTH;
    } else {
      returnVal = replyLen - 5;
    }
  } else {
    //All other protocols have 3 appended bytes
    if (replyLen < 3){
      returnVal = ERR_MSG_UNDERLENGTH;
    } else {
      returnVal = replyLen - 3;
    }
  }

  return returnVal;
}

//This function supports frames with external parity bits (MIFARE Crypto1)
//Data provided is in uint16_t format, with the parity bit as bit 15 of the input
int cr95hf_rfSendFrameWaitForReplyExtParity(cr95hf_ctx_t *ctx, uint32_t flags, uint16_t const * sendBuf, uint_least16_t sendBufLen, uint16_t *recvBuf, uint_least16_t recvBufLen, uint32_t *recvSts){
  int returnVal = ERR_OK;


  if (sendBufLen > 20){
    return ERR_MSG_OVERLENGTH;
  }

  for(int i = 0; i < sendBufLen; i++){
    uint8_t parityBit = 0x00;
    if (sendBuf[i] & 0xFF00){
      //If any bits are set, map to 0x80
      parityBit = 0x80;
    }

    ctx->tempBuf[(i * 2) + 0] = sendBuf[i] & 0xFF;
    ctx->tempBuf[(i * 2) + 1] = parityBit;

    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "TX 16b word: 0x%04X Data: 0x%02X Parity: 0x%02X", sendBuf[i], ctx->tempBuf[(i * 2) + 0], ctx->tempBuf[(i * 2) + 1]);
  }

  flags |= RFID_GENERIC_FLAG_NO_PARITY;

  int sendSts = cr95hf_rfSendFrameWaitForReply(ctx, flags, ctx->tempBuf, sendBufLen * 2, ctx->tempBuf, CR95HF_TEMP_BUF_SIZE, recvSts);

  if (sendSts >= 2){
    int numberWords = sendSts >> 1;
    if (numberWords > recvBufLen){
      syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_ERROR, "Too many words for recvBuf, got %d but buffer size = %d", numberWords, recvBufLen);
      returnVal = ERR_MSG_OVERLENGTH;
      goto end;
    }

    syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_INFO, "Sts: %d recv: 0x%08X", sendSts, *recvSts);
    //Reconstruct the frame, including the parity bits
    // round the length down, so as to skip over the ending byte count number
    for(int i = 0; i < numberWords; i++){
      recvBuf[i] = (ctx->tempBuf[(i * 2) + 0]) | (ctx->tempBuf[(i * 2) + 1] << 8);
      syslog_logFnName(CR95HF_LOG_CTX, SYSLOG_LEVEL_VERBOSE, "RX 16b word: 0x%04X Upper: 0x%02X Lower: 0x%02X", recvBuf[i], ctx->tempBuf[(i * 2) + 0], ctx->tempBuf[(i * 2) + 1]);
    }
    return sendSts / 2;
  }

end:
  return returnVal;
}

//This is really slow - should use cmd 0x07,
// but works OK enough for now.
int cr95hf_setFieldPower(cr95hf_ctx_t * ctx, bool fieldOn){
  if (fieldOn){
    return cr95hf_init(ctx, ctx->protocol);
  } else {
    return cr95hf_init(ctx, RFID_GENERIC_PROTO_OFF);
  }
}
