//NOTE:
// This file is intended as a guide to using the Mifare Classic functions only,
// and likely won't compile as-is.
//
//(C) TechnicallyObsolete 2025
#include "rf/rfid/cr95hf.h"
#include "rf/rfid/iso14443a.h"
#include "rf/rfid/tag_mifare_classic.h"


int main(void){
  cr95hf_ctx_t rfidCtx;

  syslog_init(NULL, SYSLOG_LEVEL_VERBOSE, putch);
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Syslog OK");
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "RFID LL init: %d", cr95hf_ll_init(&rfidCtx));

  cr95hf_init(&rfidCtx, RFID_GENERIC_PROTO_ISO14443A);
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "All init done");

  int sts = cr95hf_checkId(&rfidCtx);
  if (sts == ERR_OK){
    gpio_pinHigh(PLATFORM_LED_PORT, PLATFORM_LED_PIN);
    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Init OK");
  } else {
    gpio_pinLow(PLATFORM_LED_PORT, PLATFORM_LED_PIN);
    syslog_logFnName(NULL, SYSLOG_LEVEL_ERROR, "Init Failed, sts: %d", sts);
    while(1);
  }

  uint8_t uid[10] = {0};
  uint8_t uidLen = 0;
  bool iso14443p4 = false;

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Trying select");

  int selectSts = iso14443a_select((rfidGeneric_fn_sendFrameWaitForReply_t) cr95hf_rfSendFrameWaitForReply, (void *) &rfidCtx, 1, uid, &uidLen, &iso14443p4);
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Select: %d UID len: %d", selectSts, uidLen);
  for(int i= 0; i < uidLen; i++){
    xprintf("%02X ", uid[i]);
  }
  xprintf("\n");

  uint32_t uidWord = bitUtils_bufferToUint32(uid, BIT_UTILS_BIG_ENDIAN);

  tagMifareClassic_ctx_t tag = {0};
  tagMifareClassic_init(&tag, (void *) &rfidCtx, (rfidGeneric_fn_sendFrameWaitForReplyExtParity_t) cr95hf_rfSendFrameWaitForReplyExtParity);

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Trying auth");

  int authSts = -1;
  uint32_t readerChallenge = 0x99999999;
  uint8_t blockAddr = 4;

  authSts = tagMifareClassic_authBlock(&tag, 0xFFFFFFFFFFFF, TAG_MIFARE_CLASSIC_KEY_A, readerChallenge, uidWord, blockAddr);

  if (authSts == ERR_OK){
    uint8_t dataToWrite[16] = {
      0x01, 0x02, 0x03, 0x04,
      0x05, 0x06, 0x07, 0x08,
      0x09, 0x0A, 0x0B, 0x0C,
      0x0D, 0x0E, 0x0F, 0x10
    };
    int wrSts = tagMifareClassic_safeBlockWrite(&tag, blockAddr, dataToWrite);
    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Write status: %d", wrSts);


    uint8_t dataBack[16] = {0};

    int rdSts = tagMifareClassic_blockRead(&tag, blockAddr, dataBack);

    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Data from block %d (0x%02X) rd1 sts: %d:", blockAddr, blockAddr, rdSts);
    for(int i = 0; i < 16; i++){
      xprintf("%02X ", dataBack[i]);
    }
    xprintf("\n");
  }

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Read complete, shutting down RF");
  cr95hf_setFieldPower(&rfidCtx, 0);
  gpio_pinLow(PLATFORM_LED_PORT, PLATFORM_LED_PIN);

  while (1){
    delay_ms(500);
  }

}

extern cr95hf_ctx_t cr95hfCtx;

int cr95hf_ll_csLow(cr95hf_ctx_t *ctx){
  (void) ctx;
#ifdef CR95HF_UART
  //Nothing to do
#else
  delay_us(10);
  gpio_pinLow(GPIOA, 4);
  delay_us(10);
#endif
  return 0;
}

int cr95hf_ll_csHigh(cr95hf_ctx_t *ctx){
  (void) ctx;
#ifdef CR95HF_UART

#else
  while(SPI1->SR & SPI_SR_BSY){
  }


  delay_us(10);
  gpio_pinHigh(GPIOA, 4);
  delay_us(10);
#endif
  return 0;
}

int cr95hf_ll_tx(cr95hf_ctx_t *ctx, uint8_t const * data, uint_least16_t len){
  (void) ctx;
#ifdef CR95HF_VERBOSE  
  xprintf("TX buffer: %d bytes:", len);
  for(uint_least16_t i = 0; i < len; i++){
    xprintf("%02X ", data[i]);
  }
  xprintf("\r\n");
#endif

#ifdef CR95HF_UART
  for(uint_least16_t i = 0; i < len; i++){
    hwuart_tx_char(USART2, data[i]);
  }

#else
  hwspi_tx_bulk(SPI1, data, len);
#endif
  return len;
}

int cr95hf_ll_rx(cr95hf_ctx_t *ctx, uint8_t *data, uint_least16_t len){
  (void) ctx;

#ifdef CR95HF_UART
  for(uint_least16_t i = 0; i < len; i++){
    gpio_pinHigh(PLATFORM_LED_PORT, PLATFORM_LED_PIN);
    for(uint_least16_t timeout = 0; timeout < 50000; timeout++){

      if (USART2->SR & USART_SR_RXNE){
        data[i] = USART2->DR;
        gpio_pinLow(PLATFORM_LED_PORT, PLATFORM_LED_PIN);
        break;
      }
      delay_us(10);
    }
  }
#else
  hwspi_rx_bulk(SPI1, data, len, 0xFF);
#endif

#ifdef CR95HF_VERBOSE
  xprintf("RX buffer: %d bytes:", len);
  for(uint_least16_t i = 0; i < len; i++){
    xprintf("%02X ", data[i]);
  }
  xprintf("\r\n");
#endif
  return len;
}

int cr95hf_ll_init(cr95hf_ctx_t *ctx){
  (void) ctx;
  //Interface pin (GPIOB1) = 1 for SPI
  RCC->APB2ENR |= RCC_APB2ENR_IOPBEN | RCC_APB2ENR_IOPAEN;

  gpio_setPullup(GPIOA, 6, GPIO_PULL_DOWN);

  //Interface 1 = SPI
  gpio_setOutput(GPIOB, 1);
  delay_ms(10);
#ifdef CR95HF_UART
  gpio_pinHigh(GPIOB, 0);
#else
  gpio_pinHigh(GPIOB, 1);


  //Generate a pulse to latch the CR95HF into SPI mode
  gpio_setOutput(GPIOA, 2);
  gpio_pinHigh(GPIOA, 2);
  delay_ms(10);
  gpio_pinLow(GPIOA, 2);
  delay_ms(10);
  gpio_pinHigh(GPIOA, 2);
  delay_ms(10);
#endif
  delay_ms(100);
  //LED
  gpio_setOutput(GPIOB, 5);

  //CS PA4
  gpio_setOutput(GPIOA, 4);
  cr95hf_ll_csHigh(0);

#ifdef CR95HF_UART
  RCC->APB1ENR |= RCC_APB1ENR_USART2EN;
  gpio_setInput(COM1_PORT, COM1_RX_PIN);
  gpio_setPullup(COM1_PORT, COM1_RX_PIN, GPIO_PULL_UP);

  gpio_setAF(COM1_PORT, COM1_TX_PIN, 0);
//  gpio_setAF(COM1_PORT, COM1_RX_PIN, 0);
  hwuart_init(USART2, 57600);
  delay_ms(10);
  for(uint_least8_t i = 0; i < 50; i++){
    hwuart_tx_char(USART2, 0x55);
    delay_ms(1);
    if (USART2->SR & USART_SR_RXNE){
      uint8_t charIn = USART2->DR;
      if (charIn == 0x55){
        //Got a response!
        break;
      }
    }
  }
#else
  hwspi_init(48);
#endif

#ifdef CR95HF_UART
#else
  cr95hf_ll_csLow(0);
  hwspi_transfer(SPI1, CR95HF_SPI_RESET);

  cr95hf_ll_csHigh(0);
  delay_ms(100);
  delay_ms(10);

  gpio_pinLow(GPIOA, 2);
  delay_ms(10);
  gpio_pinHigh(GPIOA, 2);
  delay_ms(10);
#endif

#ifdef CR95HF_UART
  ctx->isUart = true;
#else
  ctx->isUart = false;
#endif

  return ERR_OK;
}

