#ifndef PLATFORM_DCC_LOCO_DECODER_H
#define PLATFORM_DCC_LOCO_DECODER_H

#include <stdint.h>
#include <stdbool.h>
#include "dsp/pid.h"
#include "interface/train/dcc.h"
#include "interface/train/dcc_rx.h"


//MCU: STM32L431CCU6, 256kB Flash, 64kB RAM


#define MOTOR_PWM_PORT GPIOB
#define MOTOR_PWM_A_PIN 14 //T15_CH1
#define MOTOR_PWM_B_PIN 15 //T15_CH2
#define MOTOR_PWM_AF 14



#define CS_FLASH_PORT GPIOB
#define CS_FLASH_PIN 5
#define SPI_SCK_PORT GPIOB
#define SPI_SCK_PIN 8
#define SPI_SDO_PORT GPIOB
#define SPI_SDO_PIN 9
#define SPI_SDI_PORT GPIOC
#define SPI_SDI_PIN 14

#define RAILCOM_TX1_PORT GPIOA
#define RAILCOM_TX1_PIN 9
#define RAILCOM_TX1_AF 7
#define RAILCOM_TX1_UART USART1

#define RAILCOM_TX2_PORT GPIOA
#define RAILCOM_TX2_PIN 2
#define RAILCOM_TX2_AF 7
#define RAILCOM_TX2_UART USART2

#define LINK_BUS_PORT GPIOB
#define LINK_BUS_O_PIN 3
#define LINK_BUS_I_PIN 4
#define LINK_BUS_C_PIN 10
#define LINK_BUS_C_AF 7
#define LINK_BUS_UART USART3


//PWM supported on F0F, F0R, F2
#define FN_0F_PORT GPIOB
#define FN_0F_PIN 6
#define FN_0F_AF 14 //T16_CH1N

#define FN_0R_PORT GPIOA
#define FN_0R_PIN 10
#define FN_0R_AF 1 //T1_CH3, Might not be usable if using T1_CH1 for DCC input

#define FN_1_PORT GPIOB
#define FN_1_PIN 2

#define FN_2_PORT GPIOA
#define FN_2_PIN 15
#define FN_2_AF //T2_CH1, might not be usable if using T2_CH4 for DCC input

#define FN_3_PORT GPIOB
#define FN_3_PIN 7

#define FN_4_PORT GPIOB
#define FN_4_PIN 13

#define FN_SUSI_PORT GPIOA
#define FN_SUSI_CLK_PIN 12
#define FN_SUSI_DAT_PIN 11

#define FN_INPUT1_PORT GPIOA
#define FN_INPUT1_PIN 8 //LPTIM2_OUT, T1_CH1

#define FN_MAX 5


#define MOTOR_SLEEP_PORT GPIOB
#define MOTOR_SLEEP_PIN 11
#define MOTOR_FAULT_PORT GPIOC
#define MOTOR_FAULT_PIN 15

#define MOTOR_BEMF_1_PORT GPIOA
#define MOTOR_BEMF_1_PIN 6
#define MOTOR_BEMF_1_ADC 11
#define MOTOR_BEMF_2_PORT GPIOA
#define MOTOR_BEMF_2_PIN 7
#define MOTOR_BEMF_2_ADC 12

#define MOTOR_IMOT_SENSE_PORT GPIOB
#define MOTOR_IMOT_SENSE_PIN 0
#define MOTOR_IMOT_SENSE_ADC 15
#define MOTOR_DAC_VREF_PORT GPIOA
#define MOTOR_DAC_VREF_PIN 4
#define MOTOR_DAC_VREF_CHANNEL 1

#define VBUS_SENSE_PORT GPIOB
#define VBUS_SENSE_PIN 1
#define VBUS_SENSE_ADC 16

#define KEEP_ALIVE_EN_PORT GPIOB
#define KEEP_ALIVE_EN_PIN 12 //Also T1_CH3N



//DCC Track Pins
//COMP1 -> T1_CH1 (TIM1_OR1)
//COMP2 -> T2_CH4 (TIM2_OR1)

#define TRK_PORT GPIOA
#define TRK_1_A_PIN 1
#define TRK_1_A_ADC 5 //Also COMP1_P
#define TRK_1_B_PIN 0
#define TRK_1_B_ADC 6 //Also COMP1_N, T2_CH2
#define TRK_2_A_PIN 3
#define TRK_2_A_ADC 8 //Also COMP2_P
#define TRK_2_B_PIN 5
#define TRK_2_B_ADC 10 //Also COMP2_N, T2_CH1

//CLK_CPU = 80MHz (
//Timer clock = 80 / 16 = 5MHz 
//Set ARR = 16 bits, wraps at 76Hz

//Set the lower valid '1' threshold at 52us *0.95 -> 49.4us (247)
//and upper valid '1' threshold at 64us * 1.05 -> 67.2us (336)
//
//Lower valid '0' threshold at 95us *0.95 -> 90.25us (451)
//Upper valid '0' threshold at 10000us *1.05 -> 10500us (52500)

#define PLATFORM_DCC_TIMER_PRESCALER 16
#define PLATFORM_DCC_TIMER_MIN_1 247
#define PLATFORM_DCC_TIMER_MAX_1 336
#define PLATFORM_DCC_TIMER_MIN_0 451
#define PLATFORM_DCC_TIMER_MAX_0 52500




//Flash is 2MByte
//MCU has 256kB flash, so need to reserve 256kB for a bootloader plus 256kB for update
//0x00_0000 to 0x03_FFFF (sector 0 to 63, blocks 0 to 3) is locked.

#define SPI_FLASH_PENDING_BOOT_BLANK            0xFF
#define SPI_FLASH_PENDING_BOOT_NEW_IMG_RDY      0x7F
#define SPI_FLASH_PENDING_BOOT_NEW_IMG_CONFIRM  0x3F
//Boot try 1 -> 0x1F
//Boot try 2 -> 0x0F
//Boot try 3 -> 0x07
//If any value 0x02 -> 0x06 is seen, then boot has failed
#define SPI_FLASH_PENDING_BOOT_NEW_IMG_OK       0x00

#define SPI_FLASH_EXPECTED_ID         0x0B4015
#define SPI_FLASH_WP_CMP_VAL                0
#define SPI_FLASH_WP_BP_VAL               0xB

#define SPI_FLASH_OFFSET_PENDING_BOOT       12

#define SPI_FLASH_ADDR_BL_START           0x000000  //Sector 0,   240kB for "recovery image"
#define SPI_FLASH_ADDR_FACTORY_START      0x03C000  //Sector 56,   16kB for factory data
#define SPI_FLASH_ADDR_UPDATE_START       0x040000  //Sector 64,  256kB for "update image"
#define SPI_FLASH_ADDR_CONFIG_START       0x080000  //Sector 128 -> 495 (368 sectors) available for config (slow changed vars)
#define SPI_FLASH_ADDR_CONFIG_FREQ_START  0x1F0000  //Sector 496 -> 511 (16 sectors) available for fast changed vars

#define SPI_FLASH_ADDR_END            0x1FFFFF  //Sector 511
#define SPI_FLASH_PAGE_SIZE               4096

#define BOOTLOADER_LEN_APPLICATION_MIN    512
#define BOOTLOADER_LEN_APPLICATION_MAX    0x1FF000

#define BOOTLOADER_HW_ID 0x0031

#define BOOTLOADER_ADDR_FLASH_TARGET        0x08004000
#define BOOTLOADER_ADDR_FLASH_END           0x083FFFFC
#define BOOTLOADER_ADDR_RAM_START           0x20000000
#define BOOTLOADER_ADDR_RAM_END             0x2000BFFC
#define BOOTLOADER_PAGE_SHIFT               11 //Each page is 2048 bytes, 2^11
#define BOOTLOADER_PAGE_SIZE                (1 << BOOTLOADER_PAGE_SHIFT)
#define BOOTLOADER_PAGE_MASK                ((1 << BOOTLOADER_PAGE_SHIFT) - 1)


//Backup Domain usage
//32 registers available (each 32 bits)
//[0] MAGIC1
//[1] MAGIC2
//[2] MAGIC3
//[3] MAGIC4
//[4] Function State
//[5] Motor State (7..0), target speed (15..8)
//[6] Mode flags
//[7] Last DCC Packet for this device
//[8] This node's DCC address

//[9] PID KP
//[10] PID KI
//[11] PID KD
//[12] PID eP
//[13] PID eI
//[14] PID eD
//[15] PID Setpoint

//[31] Checksum of the data words 4-8?

#define PLATFORM_BACKUP_MAGIC1 0xA1B2C3D4
#define PLATFORM_BACKUP_MAGIC2 0x04152637
#define PLATFORM_BACKUP_MAGIC3 0x55555555
#define PLATFORM_BACKUP_MAGIC4 0xAAAAAAAA

#define PLATFORM_BACKUP_ADDR_FN 4
#define PLATFORM_BACKUP_ADDR_MODE 6
#define PLATFORM_BACKUP_ADDR_LAST_PKT 7

#define PLATFORM_MODE_FLAG_DC   (1 << 0)
#define PLATFORM_MODE_FLAG_DCC  (1 << 1)

#define PLATFORM_DCC_LOCO_DECODER_ADC_MAX 10
typedef enum {
  PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_1  = 0,
  PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_2  = 1,
  PLATFORM_DCC_LOCO_DECODER_ADC_VBUS    = 2,
  PLATFORM_DCC_LOCO_DECODER_ADC_IMOT    = 3,
  PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_A = 4,
  PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_B = 5,
  PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_A = 6,
  PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_B = 7,
  PLATFORM_DCC_LOCO_DECODER_ADC_TEMP    = 8,
  PLATFORM_DCC_LOCO_DECODER_ADC_VCC     = 9
} platform_dccLocoDecoder_adcCh_e;


typedef struct {
  dccRx_ctx_t dccRx1;
  dccRx_ctx_t dccRx2;
  pid_ctx_t pid;
  uint32_t mode;
  uint32_t fnState;
  uint16_t motorTimArr;
  uint16_t motorPwmFreq;
  int16_t currentMotorSpeed;
  uint16_t motorChangeProgress;
  volatile uint16_t *adcData;
} platform_dccLocoDecoder_ctx_t;
extern platform_dccLocoDecoder_ctx_t platCtx;

void platform_initPinsAll(platform_dccLocoDecoder_ctx_t *ctx);


void platform_initPinsDccRx(void);
void platform_initPinsRailcom(void);
void platform_initPinsMotor(void);
void platform_initPinsSpi(void);
void platform_initPinsFn(void);

int platform_init(platform_dccLocoDecoder_ctx_t *ctx);
int platform_initDccTimer(platform_dccLocoDecoder_ctx_t *ctx);

int platform_initMotorPwm(platform_dccLocoDecoder_ctx_t *ctx, uint16_t pwmFreq);
int platform_initAdc(platform_dccLocoDecoder_ctx_t *ctx);

void platform_initBcmTimer(void);

int platform_getRecoveryPin(platform_dccLocoDecoder_ctx_t *ctx);
int platform_getMotorFaultStatus(void);

int platform_setMotorSpeedDir(platform_dccLocoDecoder_ctx_t *ctx, int16_t direction);
int platform_setMotorCurrentLimit(platform_dccLocoDecoder_ctx_t *ctx, uint16_t limit);
int platform_setFn(platform_dccLocoDecoder_ctx_t *ctx, uint32_t fnVal, uint32_t fnMask);
int platform_setFnIndividual(platform_dccLocoDecoder_ctx_t *ctx, uint_least8_t fn, bool en);
int platform_setBcmOutput(uint8_t output, uint8_t value);

int platform_backupDomainSave(platform_dccLocoDecoder_ctx_t *ctx);
int platform_backupDomainRestore(platform_dccLocoDecoder_ctx_t *ctx);

int platform_taskUpdateFunctionOutputs(platform_dccLocoDecoder_ctx_t *ctx);
int platform_taskAdcMeasureStart(platform_dccLocoDecoder_ctx_t *ctx, bool bemfMeasure);

int platform_getAdcVoltageFromData(platform_dccLocoDecoder_adcCh_e ch, uint16_t in);

char const *platform_getStrAdcChName(platform_dccLocoDecoder_adcCh_e ch);

#endif //PLATFORM_DCC_LOCO_DECODER_H
