#include "platform_dcc_loco_decoder.h"
#include "error_codes.h"
#include "gpio.h"
#include "delay.h"
#include "stm32l4xx.h"

#include "utils/syslog.h"
#include "storage/spi_flash.h"

#include "interface/train/railcom.h"


#include <stddef.h>

volatile uint16_t platform_adcData[PLATFORM_DCC_LOCO_DECODER_ADC_MAX];

//Supply voltages
#define VDD_CALIB 330
//Calibration value 30C
static uint16_t stm32_internalTemp_tsCal1;
//Calibration value 110C
static uint16_t stm32_internalTemp_tsCal2;

void platform_initPinsAll(platform_dccLocoDecoder_ctx_t *ctx){
  RCC->AHB2ENR |= RCC_AHB2ENR_GPIOAEN | RCC_AHB2ENR_GPIOBEN | RCC_AHB2ENR_GPIOCEN;

  platform_initPinsDccRx();
  platform_initPinsRailcom();
  platform_initPinsMotor();
  platform_initPinsFn();

  platform_initPinsSpi();

}

void platform_initPinsSpi(void){
  gpio_setOutput(SPI_SCK_PORT, SPI_SCK_PIN);
  gpio_setOutput(SPI_SDO_PORT, SPI_SDO_PIN);
  gpio_setInput(SPI_SDI_PORT, SPI_SDI_PIN);
  gpio_setPullup(SPI_SDI_PORT, SPI_SDI_PIN, GPIO_PULL_UP);

  gpio_setOutput(CS_FLASH_PORT, CS_FLASH_PIN);
  gpio_pinHigh(CS_FLASH_PORT, CS_FLASH_PIN);

  //SPI SCK approx 10MHz, need High Speed IO
  gpio_setSpeedType(CS_FLASH_PORT, CS_FLASH_PIN, GPIO_OSPEED_H, GPIO_TYPE_PUSHPULL);
  gpio_setSpeedType(SPI_SDO_PORT, SPI_SDO_PIN, GPIO_OSPEED_H, GPIO_TYPE_PUSHPULL);
  gpio_setSpeedType(SPI_SCK_PORT, SPI_SCK_PIN, GPIO_OSPEED_H, GPIO_TYPE_PUSHPULL);
}

void platform_initPinsRailcom(void){
  gpio_setAF(RAILCOM_TX1_PORT, RAILCOM_TX1_PIN, RAILCOM_TX1_AF);
  gpio_setAF(RAILCOM_TX2_PORT, RAILCOM_TX2_PIN, RAILCOM_TX2_AF);
}

void platform_initPinsDccRx(void){
  gpio_setAnalogue(TRK_PORT, TRK_1_A_PIN);
  gpio_setAnalogue(TRK_PORT, TRK_1_B_PIN);
  gpio_setAnalogue(TRK_PORT, TRK_2_A_PIN);
  gpio_setAnalogue(TRK_PORT, TRK_2_B_PIN);
}

void platform_initPinsMotor(void){
  gpio_setAnalogue(MOTOR_BEMF_1_PORT, MOTOR_BEMF_1_PIN);
  gpio_setAnalogue(MOTOR_BEMF_2_PORT, MOTOR_BEMF_2_PIN);
  gpio_setAnalogue(MOTOR_IMOT_SENSE_PORT, MOTOR_IMOT_SENSE_PIN);
  gpio_setAnalogue(VBUS_SENSE_PORT, VBUS_SENSE_PIN);

  gpio_setAF(MOTOR_PWM_PORT, MOTOR_PWM_A_PIN, MOTOR_PWM_AF);
  gpio_setAF(MOTOR_PWM_PORT, MOTOR_PWM_B_PIN, MOTOR_PWM_AF);

  gpio_setOutput(MOTOR_SLEEP_PORT, MOTOR_SLEEP_PIN);
  gpio_pinLow(MOTOR_SLEEP_PORT, MOTOR_SLEEP_PIN);

}

void platform_initPinsFn(void){
  gpio_setOutput(FN_0F_PORT, FN_0F_PIN);
  gpio_setOutput(FN_0R_PORT, FN_0R_PIN);

  gpio_setOutput(FN_1_PORT, FN_1_PIN);
  gpio_setOutput(FN_2_PORT, FN_2_PIN);
  gpio_setOutput(FN_3_PORT, FN_3_PIN);
  gpio_setOutput(FN_4_PORT, FN_4_PIN);

  //gpio_setOutput(FN_SUSI_PORT, FN_SUSI_CLK_PIN);
  //gpio_setOutput(FN_SUSI_PORT, FN_SUSI_DAT_PIN);

  gpio_setInput(FN_INPUT1_PORT, FN_INPUT1_PIN);
  gpio_setPullup(FN_INPUT1_PORT, FN_INPUT1_PIN, GPIO_PULL_UP);

}

int platform_init(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  platform_initPinsAll(ctx);
  ctx->fnState = 0;
  ctx->currentMotorSpeed = 0;
  ctx->motorChangeProgress = 0;

  //Check (and try to load if valid) the backup domain

  RCC->APB1ENR1 |= RCC_APB1ENR1_PWREN;
  platform_backupDomainRestore(ctx);

#ifdef PLATFORM_DCC_LOCO_DECODER_CMDSTN
#else
  dccRx_inputReset(&ctx->dccRx1);
  dccRx_inputReset(&ctx->dccRx2);
  platform_initDccTimer(ctx);
#endif

  ctx->adcData = platform_adcData;

  for(int i = 0; i < PLATFORM_DCC_LOCO_DECODER_ADC_MAX; i++){
    platform_adcData[i] = 0;
  }

  //Init the railcom outputs
  RCC->APB2ENR |= RCC_APB2ENR_USART1EN;   //Railcom 1 TX
  RCC->APB1ENR1 |= RCC_APB1ENR1_USART2EN; //Railcom 2 TX

  hwuart_init(RAILCOM_TX1_UART, RAILCOM_BAUD);
  hwuart_init(RAILCOM_TX2_UART, RAILCOM_BAUD);
  return ERR_OK;
}


int platform_initAdc(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (ctx->adcData == NULL){
    return ERR_BAD_PARAM;
  }

  //Load the temperature sensor calibration data

  stm32_internalTemp_tsCal1 = *((uint16_t *) 0x1FFF75A8);
  stm32_internalTemp_tsCal2 = *((uint16_t *) 0x1FFF75CA);
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "TS Cal 1: 0x%04X 2: 0x%04X", stm32_internalTemp_tsCal1, stm32_internalTemp_tsCal2);

  //ADC runs at 1MSPS, divided by 8 channels -> 125kSPS / channel
  //Most of these samples are ignored, though
  //Use DMA2

  //ADC sequence is always the same
  //
  //But only consider BEMF1 / 2 valid if a BEMF measurement is run,
  //And only consider IMOT valid if a BEMF measurement is not run
  //
  //PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_1  = 0,
  //PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_2  = 1,
  //PLATFORM_DCC_LOCO_DECODER_ADC_VBUS    = 2,
  //PLATFORM_DCC_LOCO_DECODER_ADC_IMOT    = 3,
  //PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_A = 4,
  //PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_B = 5,
  //PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_A = 6,
  //PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_B = 7,
  RCC->AHB2ENR |= RCC_AHB2ENR_ADCEN;
  RCC->AHB1ENR |= RCC_AHB1ENR_DMA1EN;

  RCC->CCIPR &= ~RCC_CCIPR_ADCSEL;
  RCC->CCIPR |= (3 << RCC_CCIPR_ADCSEL_Pos); //System clock -> ADC clock

  ADC1_COMMON->CCR &= ~(ADC_CCR_PRESC);
  ADC1_COMMON->CCR |= 2 << ADC_CCR_PRESC_Pos; //Divide by 4 -> 20MHz

  //Power up steps:
  //1: Disable deep-power-down
  ADC1->ISR = ADC_ISR_ADRDY;
  ADC1->CR &= ~ADC_CR_DEEPPWD;
  delay_ms(10);
  ADC1->CR |= ADC_CR_ADVREGEN;
  delay_ms(10);

  ADC1->CR |= ADC_CR_ADCAL;
  while(ADC1->CR & ADC_CR_ADCAL){
  }

  //Note, need at least 4 ADC clock cycle delay after ADCAL goes low
  delay_us(10);


  ADC1->CR |= ADC_CR_ADEN;
  while(!(ADC1->ISR & ADC_ISR_ADRDY)){
  }
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "ADC Ready complete");

  //Enable Temperature sensor and internal reference
  ADC1_COMMON->CCR |= ADC_CCR_TSEN | ADC_CCR_VREFEN;

  ADC1->SQR3 = (0 << ADC_SQR3_SQ10_Pos); //VREFint

  ADC1->SQR2 = (17 << ADC_SQR2_SQ9_Pos) | //Temp Sensor
    (TRK_2_B_ADC << ADC_SQR2_SQ8_Pos) | (TRK_2_A_ADC << ADC_SQR2_SQ7_Pos) |
    (TRK_1_B_ADC << ADC_SQR2_SQ6_Pos) | (TRK_1_A_ADC << ADC_SQR2_SQ5_Pos);

  ADC1->SQR1 = 
     (MOTOR_IMOT_SENSE_ADC <<  ADC_SQR1_SQ4_Pos) | (VBUS_SENSE_ADC << ADC_SQR1_SQ3_Pos) |
     (MOTOR_BEMF_2_ADC << ADC_SQR1_SQ2_Pos) | (MOTOR_BEMF_1_ADC << ADC_SQR1_SQ1_Pos) |
    (9 << ADC_SQR1_L_Pos);

  //11 channels, so ADC_SQR1_L = 10

  //Mux Selection: Mux channel 1
  DMA1_CSELR->CSELR = 0;

  //12 bit, Left aligned, DMA enabled
  ADC1->CFGR = ADC_CFGR_ALIGN | ADC_CFGR_CONT | ADC_CFGR_DMAEN | ADC_CFGR_AUTDLY;
  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "ADC init complete");
  return ERR_OK;
}

int platform_initDccTimer(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  //COMP1 -> T1_CH1 (TIM1_OR1)
  RCC->APB1ENR1 |= RCC_APB1ENR1_TIM2EN;
  RCC->APB2ENR |= RCC_APB2ENR_TIM1EN;

  TIM1->CR1 = 0;
  TIM2->CR1 = 0;

  //Timer 1:
  //Clock = 5MHz, tick = 200ns
  TIM1->PSC = (PLATFORM_DCC_TIMER_PRESCALER - 1);
  TIM1->EGR = TIM_EGR_UG;
  TIM1->ARR = 0xFFFF;

 TIM1->OR1 = TIM1_OR1_TI1_RMP; //Remap T1_IC1 to COMP1

  //Input Capture on IC1 enabled, 8 cycle deglitch filtering, capture on both edges
  TIM1->CCMR1 = (3 << TIM_CCMR1_IC1F_Pos) | (1 << TIM_CCMR1_CC1S_Pos);
  TIM1->CCER = TIM_CCER_CC1E | TIM_CCER_CC1P | TIM_CCER_CC1NP;

  TIM1->BDTR = TIM_BDTR_MOE;
  TIM1->DIER = TIM_DIER_UIE | TIM_DIER_CC1IE;
  TIM1->SR = 0;
  TIM1->CNT = 0;
  TIM1->CR1 = TIM_CR1_CEN;

  //COMP2 -> T2_CH4 (TIM2_OR1)
  TIM2->PSC = (PLATFORM_DCC_TIMER_PRESCALER - 1);
  TIM2->EGR = TIM_EGR_UG;
  TIM2->ARR = 0xFFFF;
  TIM2->OR1 = (2 << TIM2_OR1_TI4_RMP_Pos); //T2_CH4 connected to COMP2
  TIM2->CCMR2 = (3 << TIM_CCMR2_IC4F_Pos) | (1 << TIM_CCMR2_CC4S_Pos);
  TIM2->DIER = TIM_DIER_UIE | TIM_DIER_CC4IE;
  TIM2->SR = 0;
  TIM2->CNT = 0;
  TIM2->CR1 = TIM_CR1_CEN;

   //Now enable both comparators
  //Negative reference = VCC/4 -> 0.825V
  //with 23k (20k + 3k3) divider, track voltages >9V should end up as >1.1V here, so will be easily detectable.

  //COMP1 INPSEL=2 -> PA1 (TRK1_A_DIV), INMSEL=0 (VCC/4), PWRMODE=0 (High Speed)
  RCC->APB2ENR |= RCC_APB2ENR_SYSCFGEN; //Also enables the comparator


//  TRK1_B - TRK_1A mode
//  COMP1->CSR = (3 << COMP_CSR_HYST_Pos) | (2 << COMP_CSR_INPSEL_Pos) |(1 << COMP_CSR_INMESEL_Pos) | (6 << COMP_CSR_INMSEL_Pos) | (0 << COMP_CSR_PWRMODE_Pos) | (1 << COMP_CSR_EN_Pos);

// TRK_A - COMP_REF
  COMP1->CSR = (3 << COMP_CSR_HYST_Pos) | (2 << COMP_CSR_INPSEL_Pos) | COMP_CSR_SCALEN | (0 << COMP_CSR_INMSEL_Pos) | (0 << COMP_CSR_PWRMODE_Pos) | (1 << COMP_CSR_EN_Pos);

  //COMP2 INPSEL=3 -> PA5 (TRK2_B_DIV), INMSEL=0 (VCC/4), PWRMODE = 0 (High speed)
  COMP2->CSR = (3 << COMP_CSR_HYST_Pos) |(3 << COMP_CSR_INPSEL_Pos) | COMP_CSR_SCALEN | (0 << COMP_CSR_INMSEL_Pos) | (0 << COMP_CSR_PWRMODE_Pos) | (1 << COMP_CSR_EN_Pos);

  NVIC_EnableIRQ(TIM2_IRQn);
  NVIC_EnableIRQ(TIM1_CC_IRQn);
  NVIC_EnableIRQ(TIM1_UP_TIM16_IRQn);
  return ERR_OK;
}

int platform_initMotorPwm(platform_dccLocoDecoder_ctx_t *ctx, uint16_t pwmFreq){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (pwmFreq < 150){
    return ERR_BAD_PARAM;
  }

  ctx->motorPwmFreq = pwmFreq;

  gpio_setAF(MOTOR_PWM_PORT, MOTOR_PWM_A_PIN, MOTOR_PWM_AF);
  gpio_setAF(MOTOR_PWM_PORT, MOTOR_PWM_B_PIN, MOTOR_PWM_AF);

  RCC->APB2ENR |= RCC_APB2ENR_TIM15EN;
  TIM15->CR1 = 0;

  //System clock freq = 75MHz
  //Compute prescaler values
  if (pwmFreq < 500){
    TIM15->PSC = (8 - 1);
    ctx->motorTimArr = (SystemCoreClock / 8 / pwmFreq) - 1;
    //Min freq in this case 143Hz

  } else if (pwmFreq < 750){
    TIM15->PSC = (2 - 1);
    //Min Freq in this case 572Hz
    ctx->motorTimArr = (SystemCoreClock / 2 / pwmFreq) - 1;
  } else {
    TIM15->PSC = (1 - 1);
    //Min freq in this case 1145Hz
    ctx->motorTimArr = (SystemCoreClock / pwmFreq) - 1;
  }

  TIM15->EGR = TIM_EGR_UG;
  TIM15->ARR = ctx->motorTimArr;

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "For PWM freq %dHz, ARR=0x%04X PSC=0x%04X", pwmFreq, TIM15->PSC, ctx->motorTimArr);

  //PWM Mode 1, Preload enabled (only update on counter wrap)
  TIM15->CCMR1 = (6 << TIM_CCMR1_OC2M_Pos) | (6 << TIM_CCMR1_OC1M_Pos) | TIM_CCMR1_OC2PE | TIM_CCMR1_OC1PE;

  TIM15->SR = 0;
  TIM15->CNT = 0;
  TIM15->CCR1 = 0;
  TIM15->CCR2 = 0;

  TIM15->CCER = TIM_CCER_CC1E | TIM_CCER_CC2E;
  TIM15->BDTR = TIM_BDTR_MOE;
  TIM15->CR1 = TIM_CR1_CEN;

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "PWM Freq %dHz, Prescale = %d, ARR = %d", pwmFreq, TIM15->PSC, TIM15->ARR);
  return ERR_OK;
}

int platform_backupDomainSave(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }
  uint32_t tempCR1 = PWR->CR1;
  PWR->CR1 = tempCR1 | PWR_CR1_DBP;

  //Invalidate the current data
  //(Could try to double buffer it, but I'm not sure how useful that would be)
  RTC->BKP0R = 0;
  RTC->BKP1R = 0;
  RTC->BKP2R = 0;
  RTC->BKP3R = 0;

  uint32_t toWrite[5] = {0};
  uint32_t checksum = 0;

  toWrite[0] = ctx->fnState;
  toWrite[1] = (ctx->currentMotorSpeed << 16) | (ctx->motorChangeProgress);
  toWrite[2] = ctx->mode;

  for(int i = 0; i < 5; i++){
    checksum ^= toWrite[i];
  }

  //NOTE: Write the "magic" words last!
  RTC->BKP4R = toWrite[0];
  RTC->BKP5R = toWrite[1];
  RTC->BKP6R = toWrite[2];
  RTC->BKP7R = toWrite[3];
  RTC->BKP8R = toWrite[4];

  RTC->BKP31R = checksum;

  RTC->BKP3R = PLATFORM_BACKUP_MAGIC4;
  RTC->BKP2R = PLATFORM_BACKUP_MAGIC3;
  RTC->BKP1R = PLATFORM_BACKUP_MAGIC2;
  RTC->BKP0R = PLATFORM_BACKUP_MAGIC1;

  PWR->CR1 = tempCR1;
  return ERR_OK;
}


int platform_backupDomainRestore(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (RTC->BKP0R != PLATFORM_BACKUP_MAGIC1){
    return ERR_DATAERROR;
  }

  if (RTC->BKP1R != PLATFORM_BACKUP_MAGIC2){
    return ERR_DATAERROR;
  }

  if (RTC->BKP2R != PLATFORM_BACKUP_MAGIC3){
    return ERR_DATAERROR;
  }

  if (RTC->BKP3R != PLATFORM_BACKUP_MAGIC4){
    return ERR_DATAERROR;
  }

  uint32_t checksum = 0;
  checksum = RTC->BKP4R ^ RTC->BKP5R ^ RTC->BKP6R ^ RTC->BKP7R ^ RTC->BKP8R;

  if (RTC->BKP31R != checksum){
    return ERR_BAD_CHECKSUM;
  }

  return ERR_OK;
}

int platform_setFn(platform_dccLocoDecoder_ctx_t *ctx, uint32_t fnVal, uint32_t fnMask){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  uint32_t fnStateMasked = ctx->fnState & (~fnMask);
  fnStateMasked |= fnVal;

  ctx->fnState = fnStateMasked;

  return ERR_OK;
}

//BCM for function outputs
static GPIO_TypeDef*  fnList_port[]  = {FN_0F_PORT, FN_0R_PORT, FN_1_PORT, FN_2_PORT, FN_3_PORT, FN_4_PORT};
static uint16_t fnList_pin[]       = {FN_0F_PIN, FN_0R_PIN, FN_1_PIN, FN_2_PIN, FN_3_PIN, FN_4_PIN};
static uint_least8_t bcm_currentBit;


//Prescaler = 0, Timer clock freq = 80MHz
//Target min BCM frequency is 2kHz (for bit 7), leading to 16kHz for bit 0
//80M / 16k = 5000. Worst case is 5000 * 8 = 40000
#define PLATFORM_BCM_ARR 4000

uint8_t bcm_fnOutputs[6] = {0};

int platform_setBcmOutput(uint8_t output, uint8_t value){
  if (output >= sizeof(bcm_fnOutputs)){
    return ERR_BAD_PARAM;
  }

  bcm_fnOutputs[output] = value >> 3;
  return ERR_OK;
}

void platform_initBcmTimer(void){
  RCC->APB1ENR1 |= RCC_APB1ENR1_TIM7EN;
  TIM7->CR1 = 0;

  TIM7->PSC = 0;
  TIM7->EGR = TIM_EGR_UG;
  TIM7->ARR = PLATFORM_BCM_ARR - 1;
  TIM7->DIER = TIM_DIER_UIE;

  TIM7->CNT = 0;
  TIM7->SR = 0;
  TIM7->CR1 = TIM_CR1_CEN;
  NVIC_EnableIRQ(TIM7_IRQn);
  bcm_fnOutputs[0] = 0xFF;
}

void TIM7_IRQHandler(void){
  TIM7->SR = 0;
  TIM7->CNT = 0;
  for(unsigned int i = 0; i < sizeof(bcm_fnOutputs); i++){
    if (bcm_fnOutputs[0] & (1 << bcm_currentBit)){
      gpio_pinHigh(fnList_port[i], fnList_pin[i]);
    } else {
      gpio_pinLow(fnList_port[i], fnList_pin[i]);
    }
  }


  //Timer value = base period * i;
  TIM7->ARR = PLATFORM_BCM_ARR << bcm_currentBit;

  if (bcm_currentBit >= 5){
    bcm_currentBit = 0;
  } else {
    bcm_currentBit++;
  }

}

#if 0
int platform_taskUpdateFunctionOutputs(platform_dccLocoDecoder_ctx_t *ctx){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  if (ctx->fnState & (1 << 0)){
    //Directional Function is more complex
    if (ctx->currentMotorSpeed < 0){
      gpio_pinHigh(FN_0R_PORT, FN_0R_PIN);
      gpio_pinLow(FN_0F_PORT, FN_0F_PIN);
    } else {
      gpio_pinHigh(FN_0F_PORT, FN_0F_PIN);
      gpio_pinLow(FN_0R_PORT, FN_0R_PIN);
    }
  } else {
    //Both direction lights off
    gpio_pinLow(FN_0F_PORT, FN_0F_PIN);
    gpio_pinLow(FN_0R_PORT, FN_0R_PIN);
  }

  if (ctx->fnState & (1 << 1)){
    gpio_pinHigh(FN_1_PORT, FN_1_PIN);
  } else {
    gpio_pinLow(FN_1_PORT, FN_1_PIN);
  }

  if (ctx->fnState & (1 << 2)){
    gpio_pinHigh(FN_2_PORT, FN_2_PIN);
  } else {
    gpio_pinLow(FN_2_PORT, FN_2_PIN);
  }

  if (ctx->fnState & (1 << 3)){
    gpio_pinHigh(FN_3_PORT, FN_3_PIN);
  } else {
    gpio_pinLow(FN_3_PORT, FN_3_PIN);
  }

  if (ctx->fnState & (1 << 4)){
    gpio_pinHigh(FN_4_PORT, FN_4_PIN);
  } else {
    gpio_pinLow(FN_4_PORT, FN_4_PIN);
  }
  return ERR_OK;
}
#endif
int platform_setFnIndividual(platform_dccLocoDecoder_ctx_t *ctx, uint_least8_t fn, bool en){
  if (fn > FN_MAX){
    return ERR_BAD_PARAM;
  }

  uint32_t mask = (1 << fn);
  uint32_t valToSet = 0;
  if (en){
    valToSet = mask;
  }

  return platform_setFn(ctx, valToSet, mask);
}

int platform_taskAdcMeasureStart(platform_dccLocoDecoder_ctx_t *ctx, bool bemfMeasure){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  uint16_t lastCcr1 = 0;
  uint16_t lastCcr2 = 0;

  if (bemfMeasure){
    lastCcr1 = TIM15->CCR1;
    lastCcr2 = TIM15->CCR2;

    TIM15->CCR1 = 0xFFFF;
    TIM15->CCR2 = 0xFFFF;
    delay_us(100);
  }

  //must clear CCR before writing
  DMA1_Channel1->CCR = 0;

  //Configure DMA
  //ADC1 is on DMA1 Channel 1
  DMA1_CSELR->CSELR = 0;
  DMA1_Channel1->CNDTR = 10; //PLATFORM_DCC_LOCO_DECODER_ADC_MAX;

  DMA1_Channel1->CPAR = (uint32_t) &ADC1->DR;
  DMA1_Channel1->CMAR = (uint32_t) ctx->adcData;

  //16 bit memory, 16 bit peripheral, memory increment, read from peripheral, IRQ when done
  DMA1->IFCR = DMA_IFCR_CGIF1 | DMA_IFCR_CTCIF1 | DMA_IFCR_CHTIF1 | DMA_IFCR_CTEIF1;

  DMA1_Channel1->CCR =
    (0 << DMA_CCR_MEM2MEM_Pos) | (3 << DMA_CCR_PL_Pos)    |
    (1 << DMA_CCR_MSIZE_Pos)   | (2 << DMA_CCR_PSIZE_Pos) | (1 << DMA_CCR_MINC_Pos) |
    (0 << DMA_CCR_CIRC_Pos)    | (0 << DMA_CCR_DIR_Pos)   |
    (1 << DMA_CCR_TEIE_Pos)    | (1 << DMA_CCR_HTIE_Pos)  | (1 << DMA_CCR_TCIE_Pos);// |

  uint16_t lastIf = 0xFFFF;
  uint16_t lastIsr = 0xFFFF;
  uint32_t counter = 0xFFFF;
  uint16_t lastAd = 0xFFFF;
  uint16_t thisIf = DMA1->ISR;
  uint16_t thisIsr = ADC1->ISR;
  uint16_t thisAdc = ADC1->CR;

  DMA1_Channel1->CCR |= (1 << DMA_CCR_EN_Pos);

  thisIf = DMA1->ISR;
  thisIsr = ADC1->ISR;
  thisAdc = ADC1->CR;
  __DSB();

  ADC1->ISR = 0xF;

  ADC1->CR |= ADC_CR_ADSTART;
  thisIf = DMA1->ISR;
  thisIsr = ADC1->ISR;
  thisAdc = ADC1->CR;

  while(!(DMA1->ISR & DMA_ISR_GIF1)){

    thisIf = DMA1->ISR;
    thisIsr = ADC1->ISR;
    thisAdc = ADC1->CR;

    if ((thisIf != lastIf) || (thisIsr != lastIsr) || (thisAdc != lastAd)){
      lastIf = thisIf;
      lastIsr = thisIsr;
      lastAd = thisAdc;
    }
  }


  if (bemfMeasure){
    lastCcr1 = TIM15->CCR1;
    lastCcr2 = TIM15->CCR2;

    TIM15->CCR1 = lastCcr1;
    TIM15->CCR2 = lastCcr2;
  }
  ADC1->CR |= ADC_CR_ADSTP;

  return ERR_OK;
}

/*
*Return a 1 if there is a motor fault, 0 otherwise
*/
int platform_getMotorFaultStatus(void){
  if (gpio_pinRead(MOTOR_FAULT_PORT, MOTOR_FAULT_PIN) != 1){
    return 1;
  }

  return 0;
}

int platform_getRecoveryPin(platform_dccLocoDecoder_ctx_t *ctx){
  //If the backup domain is valid, then don't go to recovery mode
  int returnVal = ERR_OK;

  if (platform_backupDomainRestore(ctx) == ERR_OK){
    returnVal = 0;
    goto end;
  }

  gpio_setInput(CS_FLASH_PORT, CS_FLASH_PIN);
  gpio_setPullup(CS_FLASH_PORT, CS_FLASH_PIN, GPIO_PULL_UP);

  delay_ms(1);

  if (gpio_pinRead(CS_FLASH_PORT, CS_FLASH_PIN) == 0){
    returnVal = 1;
  } else {
    returnVal = 0;
  }

  //Ensure pin is mapped to output for flash reads to work
  gpio_setOutput(CS_FLASH_PORT, CS_FLASH_PIN);
  gpio_pinHigh(CS_FLASH_PORT, CS_FLASH_PIN);

end:
  return returnVal;
}

int platform_setMotorSpeedDir(platform_dccLocoDecoder_ctx_t *ctx, int16_t speedDir){
  if (!ctx){
    return ERR_BAD_PARAM;
  }


  bool dirReverse = !!(speedDir & 0x8000);

  uint32_t ccr = ctx->motorTimArr * (speedDir & 0x7FFF);
  ccr /= 0x7FFF;

  if (dirReverse){
    TIM15->CCR1 = 0;
    TIM15->CCR2 = ccr;
  } else {
    TIM15->CCR1 = ccr;
    TIM15->CCR2 = 0;
  }

  //syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "For speedDir = %d, ARR = 0x%04X, CCR=0x%04X", speedDir, ctx->motorTimArr, ccr);

  return ERR_OK;
}

int platform_setMotorCurrentLimit(platform_dccLocoDecoder_ctx_t *ctx, uint16_t limitMa){
  if (!ctx){
    return ERR_BAD_PARAM;
  }

  //With Riprop = 2400, at -5% VCC (3.1V) the max current readable is 2.9A
  if (limitMa > 2900){
    return ERR_BAD_PARAM;
  }

  //2k2 resistor for IPROP
  //Itrip * A * Riprop = Vref
  //Riprop = 2400
  //A = 455e-6
  //So, for 1A:
  //1 * 450e-6 * 2400 = 1.092V
  //With 3.3V 12-bit DAC, this is 1092 / 3300 * 4095 = code 1355
  uint32_t const Riprop = 2400;
  uint32_t const vcc = 3300;

  uint64_t vrefMv = limitMa * Riprop * 455;
  vrefMv = vrefMv / 1000000ULL;

  //By this point, we've got Vref in millivolts
  //Convert it to a DAC code
  uint32_t dacCodeNumerator = vrefMv * 4095;
  dacCodeNumerator /= vcc;

  if (dacCodeNumerator > 4095){
    dacCodeNumerator = 4095;
  }

  //And apply it to the DAC
  DAC->DHR12R1 = dacCodeNumerator;
  DAC->SWTRIGR = DAC_SWTRIGR_SWTRIG1;


  return ERR_OK;
}

char const *platform_getStrAdcChName(platform_dccLocoDecoder_adcCh_e ch){
  switch(ch){
    case PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_1  :  return "BEMF_1";
    case PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_2  :  return "BEMF_2";
    case PLATFORM_DCC_LOCO_DECODER_ADC_VBUS    :  return "VBUS  ";
    case PLATFORM_DCC_LOCO_DECODER_ADC_IMOT    :  return "IMOT  ";
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_A :  return "TRK_1A";
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_B :  return "TRK_1B";
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_A :  return "TRK_2A";
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_B :  return "TRK_2B";
    case PLATFORM_DCC_LOCO_DECODER_ADC_TEMP    :  return "TEMP  ";
    case PLATFORM_DCC_LOCO_DECODER_ADC_VCC     :  return "VCC   ";
    default                                    :  return "??????";
  
  }
}

int stm32_internalTemp_getCalibrated(uint16_t in, uint16_t vcc_mv, int16_t *result){
#if 1
  vcc_mv /= 10;
  //From the STM32 application note
  int32_t temperature; /* will contain the temperature in degree Celsius */
  temperature = (((int32_t) in * vcc_mv / VDD_CALIB)- (int32_t) stm32_internalTemp_tsCal1);
  temperature = temperature * (int32_t) (110 - 30);
  temperature = temperature / (int32_t) (stm32_internalTemp_tsCal2 - stm32_internalTemp_tsCal1);
  temperature = temperature + 30;
  *result = temperature * 10;
#endif
  return 0;
}

int platform_getAdcVoltageFromData(platform_dccLocoDecoder_adcCh_e ch, uint16_t in){
  uint32_t scaleFactor = 0;

  //The following channels all have the same scale factors:
  switch(ch){
    case PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_1:
      //Fall Through
    case PLATFORM_DCC_LOCO_DECODER_ADC_BEMF_2:
      //Fall Through
    case PLATFORM_DCC_LOCO_DECODER_ADC_VBUS:
      scaleFactor = 30683;
      break;
    case PLATFORM_DCC_LOCO_DECODER_ADC_IMOT:
      //For the motor current sense:
      //gain = 455uA / A, which with a 2.37k resistor is 1.07835 V per A
      scaleFactor = 10784;
      break;
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_A :
      //Fall through
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_1_B :
      //Fall through
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_A :
      //Fall through
    case PLATFORM_DCC_LOCO_DECODER_ADC_TRK_2_B :
      scaleFactor = 30683;
      break;
    case PLATFORM_DCC_LOCO_DECODER_ADC_TEMP:
      scaleFactor = 0;
      break;
    case PLATFORM_DCC_LOCO_DECODER_ADC_VCC:
      //vrefint = 1.212V (typical)
      scaleFactor = 8700;
      break;
    default:
      scaleFactor = 0;
  }


  uint32_t temp = in * scaleFactor;
  temp /= 65520;

  if (temp > INT32_MAX){
    return INT32_MAX;
  }

  return temp;

}

//IRQ handlers
platform_dccLocoDecoder_ctx_t platCtx = {0};
uint32_t volatile t1irq;

void TIM1_UP_TIM16_IRQHandler(void){
  t1irq = 1;
  uint32_t sr = TIM1->SR;
  TIM1->SR = ~TIM_SR_UIF;

  __DSB();
}

void TIM1_CC_IRQHandler(void){

  uint32_t sr = TIM1->SR;
#ifdef PLATFORM_DCC_LOCO_DECODER_CMDSTN
#else
  if (sr & TIM_SR_CC1IF){
    uint32_t time = TIM1->CCR1;
    TIM1->CNT = 0;

    //LINK_BUS_UART->TDR = (time >> 8);
    __DSB();
    __DSB();
   // LINK_BUS_UART->TDR = (time & 0xFF);

    if ((time >= PLATFORM_DCC_TIMER_MIN_1) && (time <= PLATFORM_DCC_TIMER_MAX_1)){
      dccRx_inputTimerState(&platCtx.dccRx1, DCCRX_TIMER_STATE_SHORT);
    } else if ((time >= PLATFORM_DCC_TIMER_MIN_0) && (time <= PLATFORM_DCC_TIMER_MAX_0)){
      dccRx_inputTimerState(&platCtx.dccRx1, DCCRX_TIMER_STATE_LONG);
    } else {
      dccRx_inputTimerState(&platCtx.dccRx1, DCCRX_TIMER_STATE_ERROR);
    }
  }

  if (sr & TIM_SR_UIF){
    TIM1->SR = ~TIM_SR_UIF;
    dccRx_inputTimerState(&platCtx.dccRx1, DCCRX_TIMER_STATE_ERROR);
    //TODO - also forward this error to the "time since last RX" logic
  }
#endif
  __DSB();
}

void TIM2_IRQHandler(void){
  uint32_t sr = TIM2->SR;
#ifdef PLATFORM_DCC_LOCO_DECODER_CMDSTN
#else
  if (sr & TIM_SR_CC4IF){
    uint32_t time = TIM2->CCR4;
    //TIM2->CNT = 0;

    TIM2->SR = ~TIM_SR_CC4IF;

    if ((time >= PLATFORM_DCC_TIMER_MIN_1) && (time <= PLATFORM_DCC_TIMER_MAX_1)){
      dccRx_inputTimerState(&platCtx.dccRx2, DCCRX_TIMER_STATE_SHORT);
    } else if ((time >= PLATFORM_DCC_TIMER_MIN_0) && (time <= PLATFORM_DCC_TIMER_MAX_0)){
      dccRx_inputTimerState(&platCtx.dccRx2, DCCRX_TIMER_STATE_LONG);
    } else {
      dccRx_inputTimerState(&platCtx.dccRx2, DCCRX_TIMER_STATE_ERROR);
    }
  }

  if (sr & TIM_SR_UIF){
    TIM2->SR = ~TIM_SR_UIF;
    dccRx_inputTimerState(&platCtx.dccRx2, DCCRX_TIMER_STATE_ERROR);
    //TODO - also forward this error to the "time since last RX" logic
  }
#endif
  __DSB();
}

//Device Drivers
uint8_t platform_swSpi_transfer(uint8_t data){
  uint8_t returnVal = 0;

  for(int i = 7; i >= 0; i--){
    int thisBit = (data >> i) & 0x1;
    SPI_SDO_PORT->BSRR = thisBit ? (1 << SPI_SDO_PIN) : (1 << (16 + SPI_SDO_PIN));
    SPI_SCK_PORT->BSRR = (1 << SPI_SCK_PIN);
    if (SPI_SDI_PORT->IDR & (1 << SPI_SDI_PIN)){
      returnVal |= (1 << i);
    }
    SPI_SCK_PORT->BSRR = (1 << (16 + SPI_SCK_PIN));
  }
  return returnVal;
}

void platform_swSpi_wr(uint8_t data){

  for(int i = 7; i >= 0; i--){
    int thisBit = (data >> i) & 0x1;
    SPI_SDO_PORT->BSRR = thisBit ? (1 << SPI_SDO_PIN) : (1 << (16 + SPI_SDO_PIN));
    SPI_SCK_PORT->BSRR = (1 << SPI_SCK_PIN);
    SPI_SCK_PORT->BSRR = (1 << (16 + SPI_SCK_PIN));
  }
}

uint8_t platform_swSpi_rd(void){
  uint8_t returnVal = 0;

  for(int i = 7; i >= 0; i--){
    SPI_SCK_PORT->BSRR = (1 << SPI_SCK_PIN);
    if (SPI_SDI_PORT->IDR & (1 << SPI_SDI_PIN)){
      returnVal |= (1 << i);
    }
    SPI_SCK_PORT->BSRR = (1 << (16 + SPI_SCK_PIN));
  }
  return returnVal;

}

int spiflash_ll_csLow(spiflash_ctx_t *ctx){
  (void) ctx;
  gpio_pinLow(CS_FLASH_PORT, CS_FLASH_PIN);
  return ERR_OK;
}

int spiflash_ll_csHigh(spiflash_ctx_t *ctx){
  (void) ctx;
  gpio_pinHigh(CS_FLASH_PORT, CS_FLASH_PIN);
  return ERR_OK;
}

int spiflash_ll_init(spiflash_ctx_t *ctx){
  (void) ctx;
  gpio_setOutput(CS_FLASH_PORT, CS_FLASH_PIN);
  gpio_pinHigh(CS_FLASH_PORT, CS_FLASH_PIN);
  return ERR_OK;
}

int spiflash_ll_rdBulk(spiflash_ctx_t *ctx, uint8_t *data, uint32_t len){
  (void) ctx;
  for(uint32_t i = 0; i < len; i++){
    data[i] = platform_swSpi_rd();
  }

  return ERR_OK;
}

int spiflash_ll_wrBulk(spiflash_ctx_t *ctx, uint8_t const *data, uint32_t len){
  (void) ctx;
  for(uint32_t i = 0; i < len; i++){
    platform_swSpi_wr(data[i]);
  }
  return ERR_OK;
}


