#include "flash_dcc_loco_decoder.h"
#include "platform_dcc_loco_decoder.h"

#include "interface/train/dcc_cv.h"
#include "error_codes.h"

#include "storage/spi_flash.h"

#include "utils/syslog.h"
#include "utils/xprintf.h"
#include <string.h>

extern spiflash_ctx_t spiFlash;

static uint16_t flashDccLocoDecoder_freqModifiedCvs[] = {
  DCC_CV_M_3_ACCELERATION,
  DCC_CV_M_4_DECELERATION,
  DCC_CV_M_19_CONSIST_ADDR,
  DCC_CV_M_23_ACCELERATION_ADJ,
  DCC_CV_M_24_DECELERATION_ADJ,
  DCC_CV_M_29_CFG1,
  0x0000
};

int flashDccLocoDecoder_getFreqStoreIdx(uint16_t cv){
  int idx = 0;
  while(true){
    uint16_t thisCv = flashDccLocoDecoder_freqModifiedCvs[idx];
    if (thisCv == 0){
      return ERR_NOTFOUND;
    }

    if (thisCv == cv){
      return idx;
    }

    idx++;
  }
  return ERR_NOTFOUND;
}


int flashDccLocoDecoder_cvLoadPage(uint8_t target[FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK], uint16_t startCv){

  //Map a start CV to a page
  uint_least16_t pageN = (startCv - 1) / FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK;

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Trying to load CV %d, page = 0x%04X", startCv, pageN);

  //Within a page, try to find the last valid location
  int validLoc = ERR_NOTFOUND;

  for(unsigned int wl = 0; wl < FLASH_DCC_LOCO_DECODER_CV_WEAR_LEVEL; wl++){
    uint8_t rdBuf[FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN];

    uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_START + (pageN * FLASH_DCC_LOCO_DECODER_ERASE_BLOCK_SIZE) + (wl * FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN);
    spiflash_readBlock(&spiFlash, rdAddr, sizeof(rdBuf), rdBuf);


    //Header check
    bool headerValid = false;

    if ((rdBuf[0] == 0x55) && (rdBuf[1] == 0xAA)){
      //TODO, should be more robust...
      headerValid = true;
    }

    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Read wl=%d, addr=0x%06X valid: %d", wl, rdAddr, headerValid);

    if (headerValid){
      validLoc = wl;
      memcpy(target, rdBuf + FLASH_DCC_LOCO_DECODER_HEADER_LEN, FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK);
      //Don't break - need to find the last valid copy of the data, not the first
      //Could avoid doing the repeated memcpy, but it's fairly quick anyway
    } else {
      //Invalid header -> No more configs available
      break;
    }

  }

  //Having read the page, now overlay a fast read CV if required
  for(uint_least16_t i = 0; i < FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK; i++){
    uint16_t cv = i + startCv;

    int inFastStoreList = flashDccLocoDecoder_getFreqStoreIdx(cv);
    if (inFastStoreList >= 0){
      syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Overlay CV %d", cv);

      flashDccLocoDecoder_cvLoadFreqMod(&target[i], cv);
    }
  }


  return validLoc;
}

int flashDccLocoDecoder_cvLoadBulk(uint8_t target[256]){
  for(uint_least8_t i = 0; i < 240; i += FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK){
    flashDccLocoDecoder_cvLoadPage(target + i, i + 1);
  }
  return ERR_OK;
}

int flashDccLocoDecoder_cvLoadFreqMod(uint8_t target[1], uint16_t cv){
  int freqModIdx = flashDccLocoDecoder_getFreqStoreIdx(cv);

  if (freqModIdx < 0){
    syslog_logFnName(NULL, SYSLOG_LEVEL_ERROR, "CV %d not in Freq Mod list", cv);
    return freqModIdx;
  }

  syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Loading Freq Mod CV %d", cv);
  bool headerValid = false;

  for(unsigned int wl = 0; wl < FLASH_DCC_LOCO_DECODER_CV_WEAR_LEVEL; wl++){
    uint8_t rdBuf[FLASH_DCC_LOCO_DECODER_CONFIG_FAST_BLOCK_LEN];
    uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_FREQ_START + (freqModIdx * SPI_FLASH_PAGE_SIZE) + (wl * FLASH_DCC_LOCO_DECODER_CONFIG_FAST_BLOCK_LEN);

    spiflash_readBlock(&spiFlash, rdAddr, sizeof(rdBuf), rdBuf);


    //Header check
    if ((rdBuf[0] == 0x55) && (rdBuf[1] == 0xAA)){
      //TODO, should be more robust...
      headerValid = true;
    } else {
      headerValid = false;
    }


    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Read wl=%d, addr=0x%06X valid: %d, got 0x%02X", wl, rdAddr, headerValid, rdBuf[8]);
    if (headerValid){
      target[0] = rdBuf[8];
    } else {
      //Not found, break
      break;
    }

  }

  if (!headerValid){
    return ERR_NOTFOUND;
  }

  return ERR_OK;
}


int flashDccLocoDecoder_cvStore(uint8_t allCv[256], uint16_t cv, uint8_t newValue){


  int freqUpdateIdx = flashDccLocoDecoder_getFreqStoreIdx(cv);

  if (freqUpdateIdx >= 0){
    //It's a CV by itself, no need to merge existing data
    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Write overlay CV %d = 0x%02X", cv, newValue);

    int wrLoc = -1;
    uint8_t rdBuf[FLASH_DCC_LOCO_DECODER_CONFIG_FAST_BLOCK_LEN];
    for(unsigned int wl = 0; wl < FLASH_DCC_LOCO_DECODER_CV_WEAR_LEVEL; wl++){

      uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_FREQ_START+ (freqUpdateIdx * SPI_FLASH_PAGE_SIZE) + (wl * FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN);
      spiflash_readBlock(&spiFlash, rdAddr, sizeof(rdBuf), rdBuf);

      //Header check
      //TODO, this really should check if the flash is blank...
      bool headerValid = false;

      if ((rdBuf[0] == 0x55) && (rdBuf[1] == 0xAA)){
        headerValid = true;
      }

      if (rdBuf[8] == newValue){
        wrLoc = wl;
        syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "New value equal to old value, not writing");
        break;
      }


      if (!headerValid){
        //Found a space
        syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Found write location at WL=%d, addr=0x%08X", wl, rdAddr);
        rdBuf[0] = 0x55;
        rdBuf[1] = 0xAA;
        rdBuf[8] = newValue;
        wrLoc = wl;

        for(int i = 0; i < sizeof(rdBuf); i++){
          xprintf("%02X ", rdBuf[i]);
        }
        xprintf("\n");

        spiflash_writeBlock(&spiFlash, rdAddr, FLASH_DCC_LOCO_DECODER_CONFIG_FAST_BLOCK_LEN, rdBuf, 1);
        break;
      }


    }

    //Handle the case where all blocks in the wear level list are used
    if (wrLoc == -1){

      uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_FREQ_START+ (freqUpdateIdx * SPI_FLASH_PAGE_SIZE) + (0 * FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN);
      syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Erasing 0x%08X", rdAddr);
      spiflash_eraseSector(&spiFlash, rdAddr, 0, 1);

      //Make the most of the fact that the previous read will have populated rdBuf with all but the latest data
      rdBuf[0] = 0x55;
      rdBuf[1] = 0xAA;
      rdBuf[8] = newValue;

      spiflash_writeBlock(&spiFlash, rdAddr, FLASH_DCC_LOCO_DECODER_CONFIG_FAST_BLOCK_LEN, rdBuf, 1); 
    }
  } else {
    uint_least16_t pageN = (cv - 1) / FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK;
    uint_least16_t offsetInPage = (cv - 1) % FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK;

    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Write CV %d = 0x%02X, pageN=0x%04X offset=%d", 
      cv, newValue, pageN, offsetInPage
    );

    //Now find the write location 
    //In the event of a write, if all wear levelled locations have been used, need to do an erase cycle
    int wrLoc = -1;
    uint8_t rdBuf[FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN];


    uint8_t previousCvVal = 0;
    
    for(unsigned int wl = 0; wl < FLASH_DCC_LOCO_DECODER_CV_WEAR_LEVEL; wl++){

      uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_START + (pageN * FLASH_DCC_LOCO_DECODER_ERASE_BLOCK_SIZE) + (wl * FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN);
      spiflash_readBlock(&spiFlash, rdAddr, sizeof(rdBuf), rdBuf);

      //Header check
      //TODO, this really should check if the flash is blank...
      bool headerValid = false;

      if ((rdBuf[0] == 0x55) && (rdBuf[1] == 0xAA)){
        headerValid = true;
        previousCvVal = rdBuf[8 + offsetInPage];
      }


      if (!headerValid){
        //Found a space
        //Check to see if the write should happen
        syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Found write location at WL=%d, addr=0x%08X", wl, rdAddr);
        rdBuf[0] = 0x55;
        rdBuf[1] = 0xAA;
        rdBuf[8 + offsetInPage] = newValue;
        wrLoc = wl;


        for(int i = 0; i < sizeof(rdBuf); i++){
          xprintf("%02X ", rdBuf[i]);
        }
        xprintf("\n");

        if (previousCvVal == newValue){
          syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Last value equal, not writing");
        } else {
          spiflash_writeBlock(&spiFlash, rdAddr, FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN, rdBuf, 1);
        }
        break;
      }


    }

    //Handle the case where all blocks in the wear level list are used
    if (wrLoc == -1){

      uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_START + (pageN * FLASH_DCC_LOCO_DECODER_ERASE_BLOCK_SIZE) + (0 * FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN);
      syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Erasing 0x%08X", rdAddr);
      spiflash_eraseSector(&spiFlash, rdAddr, 0, 1);

      //Make the most of the fact that the previous read will have populated rdBuf with all but the latest data
      rdBuf[0] = 0x55;
      rdBuf[1] = 0xAA;
      rdBuf[8 + offsetInPage] = newValue;

      spiflash_writeBlock(&spiFlash, rdAddr, FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN, rdBuf, 1);

    }

  }
  return ERR_OK;
}

//TODO - flip the polarity, so '0' is stored as '1'?

int flashDccLocoDecoder_cvResetDefault(void){
  for(int cv = 1; cv <= 240; cv+= FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK){
    uint_least16_t pageN = (cv - 1) / FLASH_DCC_LOCO_DECODER_CV_PER_CONFIG_BLOCK;

    uint32_t rdAddr = SPI_FLASH_ADDR_CONFIG_START + (pageN * FLASH_DCC_LOCO_DECODER_ERASE_BLOCK_SIZE);
    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Erasing 0x%08X", rdAddr);
    spiflash_eraseSector(&spiFlash, rdAddr, 0, 1);

    uint8_t wrBuf[FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN] = {0};
    wrBuf[0] = 0x55;
    wrBuf[1] = 0xAA;
    if (cv == 1){
      wrBuf[8] = 3;
    }
    spiflash_writeBlock(&spiFlash, rdAddr, FLASH_DCC_LOCO_DECODER_CONFIG_BLOCK_LEN, wrBuf, 1);
    syslog_logFnName(NULL, SYSLOG_LEVEL_INFO, "Writing 0x%08X", rdAddr);
  }

  return ERR_OK;
}
